# ORM Optimization Patterns

## Quick Navigation
- [Django ORM](#django-orm) - select_related, prefetch_related
- [SQLAlchemy](#sqlalchemy) - joinedload, selectinload
- [ActiveRecord](#activerecord-ruby-on-rails) - includes, eager_load
- [Prisma](#prisma-typescriptjavascript) - include
- [Sequelize](#sequelize) - include
- [Bulk Operations](#bulk-operations)
- [Connection Management](#connection--transaction-management)

## Contents
- N+1 Query Solutions by Framework
- Eager Loading Strategies
- Bulk Operations
- Query Optimization Techniques
- Connection & Transaction Management

## N+1 Query Solutions by Framework

### Django ORM

**Problem:**
```python
# N+1: 1 query for users + N queries for orders
users = User.objects.all()
for user in users:
    orders = user.orders.all()  # Query per user!
```

**Solution - select_related (FK/OneToOne):**
```python
# Single JOIN query
users = User.objects.select_related('profile').all()
```

**Solution - prefetch_related (ManyToMany/Reverse FK):**
```python
# 2 queries: users + orders
users = User.objects.prefetch_related('orders').all()
```

**Solution - Prefetch with filter:**
```python
from django.db.models import Prefetch

users = User.objects.prefetch_related(
    Prefetch('orders', queryset=Order.objects.filter(status='pending'))
).all()
```

### SQLAlchemy

**Problem:**
```python
# N+1 with lazy loading
users = session.query(User).all()
for user in users:
    print(user.orders)  # Query per user!
```

**Solution - joinedload (single JOIN):**
```python
from sqlalchemy.orm import joinedload

users = session.query(User).options(joinedload(User.orders)).all()
```

**Solution - selectinload (separate IN query):**
```python
from sqlalchemy.orm import selectinload

# Better for large datasets
users = session.query(User).options(selectinload(User.orders)).all()
```

**Solution - subqueryload:**
```python
from sqlalchemy.orm import subqueryload

# Subquery instead of IN
users = session.query(User).options(subqueryload(User.orders)).all()
```

### ActiveRecord (Ruby on Rails)

**Problem:**
```ruby
# N+1
User.all.each do |user|
  puts user.orders.count  # Query per user!
end
```

**Solution - includes (preload):**
```ruby
# 2 queries
User.includes(:orders).each do |user|
  puts user.orders.count
end
```

**Solution - eager_load (LEFT JOIN):**
```ruby
# Single JOIN query
User.eager_load(:orders).where(orders: { status: 'pending' })
```

**Solution - preload (separate queries):**
```ruby
# Forces separate queries (useful when includes chooses wrong strategy)
User.preload(:orders).all
```

### Prisma (TypeScript/JavaScript)

**Problem:**
```typescript
// N+1
const users = await prisma.user.findMany();
for (const user of users) {
  const orders = await prisma.order.findMany({ where: { userId: user.id } });
}
```

**Solution - include:**
```typescript
const users = await prisma.user.findMany({
  include: { orders: true }
});
```

**Solution - include with filter:**
```typescript
const users = await prisma.user.findMany({
  include: {
    orders: {
      where: { status: 'PENDING' },
      take: 10
    }
  }
});
```

### Sequelize

**Problem:**
```javascript
// N+1
const users = await User.findAll();
for (const user of users) {
  const orders = await Order.findAll({ where: { userId: user.id } });
}
```

**Solution - include:**
```javascript
const users = await User.findAll({
  include: [{
    model: Order,
    where: { status: 'pending' },
    required: false  // LEFT JOIN
  }]
});
```

## Bulk Operations

### Django
```python
# Bulk create (single INSERT)
User.objects.bulk_create([
    User(name='Alice'),
    User(name='Bob'),
], batch_size=1000)

# Bulk update
User.objects.filter(active=False).update(status='inactive')

# Bulk update with different values
User.objects.bulk_update(users, ['name', 'email'], batch_size=1000)
```

### SQLAlchemy
```python
# Bulk insert
session.bulk_insert_mappings(User, [
    {'name': 'Alice'},
    {'name': 'Bob'},
])

# Bulk update
session.bulk_update_mappings(User, [
    {'id': 1, 'status': 'active'},
    {'id': 2, 'status': 'inactive'},
])

# Or use Core for best performance
from sqlalchemy import insert
session.execute(insert(User), [{'name': 'Alice'}, {'name': 'Bob'}])
```

### Prisma
```typescript
// Bulk create
await prisma.user.createMany({
  data: [
    { name: 'Alice' },
    { name: 'Bob' },
  ],
  skipDuplicates: true,
});

// Bulk update
await prisma.user.updateMany({
  where: { active: false },
  data: { status: 'inactive' },
});
```

## Query Optimization Techniques

### Select Only Needed Columns

**Django:**
```python
# Only specific fields
User.objects.only('id', 'name')
User.objects.values('id', 'name')  # Returns dicts
User.objects.values_list('id', 'name')  # Returns tuples
```

**SQLAlchemy:**
```python
session.query(User.id, User.name).all()
# Or with ORM objects
session.query(User).options(load_only(User.id, User.name)).all()
```

**Prisma:**
```typescript
await prisma.user.findMany({
  select: { id: true, name: true }
});
```

### Use Aggregations in Database

**Django:**
```python
from django.db.models import Count, Sum, Avg

User.objects.annotate(
    order_count=Count('orders'),
    total_spent=Sum('orders__amount')
).filter(order_count__gt=5)
```

**SQLAlchemy:**
```python
from sqlalchemy import func

session.query(
    User,
    func.count(Order.id).label('order_count')
).join(Order).group_by(User.id).all()
```

### Pagination

**Django:**
```python
# Offset pagination (slow for large offsets)
User.objects.all()[1000:1020]

# Keyset pagination (fast)
last_id = 1000
User.objects.filter(id__gt=last_id).order_by('id')[:20]
```

**SQLAlchemy:**
```python
# Keyset pagination
session.query(User).filter(User.id > last_id).order_by(User.id).limit(20)
```

## Connection & Transaction Management

### Connection Pooling

**SQLAlchemy:**
```python
from sqlalchemy import create_engine

engine = create_engine(
    'postgresql://...',
    pool_size=10,
    max_overflow=20,
    pool_pre_ping=True,  # Verify connections
    pool_recycle=3600,   # Recycle after 1 hour
)
```

**Django:**
```python
# settings.py
DATABASES = {
    'default': {
        'CONN_MAX_AGE': 600,  # Keep connections for 10 minutes
        'CONN_HEALTH_CHECKS': True,  # Django 4.1+
    }
}
```

### Transaction Boundaries

**Django:**
```python
from django.db import transaction

# Atomic block
with transaction.atomic():
    user.save()
    order.save()

# Or decorator
@transaction.atomic
def create_order(user, items):
    ...
```

**SQLAlchemy:**
```python
# Session handles transactions
with Session() as session:
    session.add(user)
    session.add(order)
    session.commit()  # Single transaction

# Nested transactions
with session.begin_nested():
    # Savepoint
    session.add(risky_operation)
```

### Avoiding Long Transactions

```python
# Bad: Long transaction
with transaction.atomic():
    for item in huge_list:
        process(item)  # Locks held for entire loop

# Good: Batch transactions
for batch in chunks(huge_list, 100):
    with transaction.atomic():
        for item in batch:
            process(item)
```
