# SQL Optimization Patterns

## Quick Navigation
- [Index Optimization](#index-optimization) - missing indexes, anti-patterns
- [Query Rewrite](#query-rewrite-patterns) - subquery→JOIN, EXISTS vs IN
- [Join Optimization](#join-optimization) - order, type selection
- [Aggregation](#aggregation-optimization) - pre-filtering, partial aggregation
- [Performance Killers](#common-performance-killers) - N+1, cartesian, type conversion
- [PostgreSQL](#postgresql) - partial indexes, CTEs, pg_stat_statements
- [MySQL](#mysql) - EXPLAIN ANALYZE, key_len, covering indexes
- [SQLite](#sqlite) - EXPLAIN QUERY PLAN, WAL mode

## Contents
- Index Optimization (missing index detection, anti-patterns)
- Query Rewrite Patterns (subquery→JOIN, EXISTS vs IN, pagination)
- Join Optimization (order, type selection)
- Aggregation Optimization (pre-filtering, partial aggregation)
- Common Performance Killers
- Database-Specific Optimizations (PostgreSQL, MySQL, SQLite)

## Index Optimization

### Missing Index Detection
Look for:
- WHERE clauses on non-indexed columns
- JOIN conditions without supporting indexes
- ORDER BY on non-indexed columns with LIMIT
- Composite index opportunities (leftmost prefix rule)

### Index Anti-patterns
- Over-indexing (too many indexes slow writes)
- Redundant indexes (prefix overlap)
- Low-cardinality indexes (boolean, status columns alone)
- Function calls on indexed columns: `WHERE UPPER(name) = 'X'` → no index use

## Query Rewrite Patterns

### Subquery → JOIN
```sql
-- Before (correlated subquery)
SELECT * FROM orders o 
WHERE customer_id IN (SELECT id FROM customers WHERE region = 'US');

-- After (JOIN)
SELECT o.* FROM orders o
INNER JOIN customers c ON o.customer_id = c.id
WHERE c.region = 'US';
```

### EXISTS vs IN
```sql
-- Use EXISTS for large subquery results
SELECT * FROM orders o
WHERE EXISTS (SELECT 1 FROM customers c WHERE c.id = o.customer_id AND c.active = 1);

-- Use IN for small, known sets
SELECT * FROM orders WHERE status IN ('pending', 'processing');
```

### Avoiding SELECT *
- Only select needed columns
- Reduces I/O and memory
- Enables covering index usage

### UNION vs UNION ALL
- Use UNION ALL when duplicates acceptable (faster, no sort)
- UNION requires sort + dedup

### Pagination Optimization
```sql
-- Bad: OFFSET becomes slow for large values
SELECT * FROM logs ORDER BY created_at DESC LIMIT 20 OFFSET 10000;

-- Better: Keyset/cursor pagination
SELECT * FROM logs 
WHERE created_at < '2024-01-01 12:00:00'
ORDER BY created_at DESC LIMIT 20;
```

## Join Optimization

### Join Order
- Smaller table first (when optimizer hints needed)
- Filter early (WHERE before JOIN when possible)
- Consider materialized subqueries for complex filters

### Join Type Selection
- INNER JOIN: Only matching rows
- LEFT JOIN: All from left + matching right (check if really needed)
- Avoid implicit cross joins

## Aggregation Optimization

### Pre-filtering
```sql
-- Bad: Filter after aggregation
SELECT customer_id, SUM(amount) as total
FROM orders GROUP BY customer_id
HAVING SUM(amount) > 1000;

-- Better: Filter before when possible
SELECT customer_id, SUM(amount) as total
FROM orders WHERE status = 'completed'
GROUP BY customer_id HAVING SUM(amount) > 1000;
```

### Partial Aggregation
For time-series, consider pre-aggregated summary tables.

## Common Performance Killers

1. **N+1 queries**: Detect loops executing individual queries
2. **Cartesian products**: Missing JOIN conditions
3. **Implicit type conversion**: `WHERE string_col = 123`
4. **OR conditions**: Often prevent index use
5. **LIKE with leading wildcard**: `LIKE '%term'`
6. **Functions in WHERE**: `WHERE YEAR(date_col) = 2024`
7. **NOT IN with NULLs**: Use NOT EXISTS instead
8. **Large IN lists**: Consider temp table + JOIN

## Database-Specific Optimizations

### PostgreSQL
- Use `EXPLAIN (ANALYZE, BUFFERS)` for real metrics
- Consider partial indexes: `CREATE INDEX ON orders(status) WHERE status = 'pending'`
- Use `pg_stat_statements` for query profiling
- CTEs are optimization fences (pre-v12)

### MySQL
- Use `EXPLAIN ANALYZE` (8.0+)
- Force index hints sparingly: `USE INDEX(idx_name)`
- Check `key_len` in EXPLAIN for partial index use
- Consider covering indexes for high-read tables

### SQLite
- Use `EXPLAIN QUERY PLAN`
- Create indexes after bulk inserts
- Use WITHOUT ROWID for specific cases
- WAL mode for concurrent reads
