#!/usr/bin/env python3
"""
Analyze SQL query complexity and detect common anti-patterns.
Usage: python analyze_query.py "SELECT * FROM orders WHERE ..."
       python analyze_query.py --file query.sql
"""

import re
import sys
import json
import argparse
from dataclasses import dataclass, asdict
from typing import List

@dataclass
class Issue:
    severity: str  # "high", "medium", "low"
    pattern: str
    description: str
    suggestion: str

@dataclass
class AnalysisResult:
    complexity_score: int  # 1-10, higher = more complex
    issues: List[Issue]
    metrics: dict

def analyze_query(sql: str) -> AnalysisResult:
    """Analyze SQL query for complexity and anti-patterns."""
    sql_upper = sql.upper()
    sql_normalized = re.sub(r'\s+', ' ', sql_upper).strip()
    
    issues = []
    metrics = {
        "tables": 0,
        "joins": 0,
        "subqueries": 0,
        "conditions": 0,
        "aggregations": 0,
    }
    
    # Count structural elements
    metrics["joins"] = len(re.findall(r'\bJOIN\b', sql_upper))
    metrics["subqueries"] = sql_upper.count('SELECT') - 1
    metrics["conditions"] = sql_upper.count(' AND ') + sql_upper.count(' OR ') + 1 if 'WHERE' in sql_upper else 0
    metrics["aggregations"] = sum(sql_upper.count(f) for f in ['COUNT(', 'SUM(', 'AVG(', 'MAX(', 'MIN(', 'GROUP BY'])
    
    # Detect FROM tables (simplified)
    from_match = re.search(r'\bFROM\b\s+(\w+)', sql_upper)
    if from_match:
        metrics["tables"] = 1 + metrics["joins"]
    
    # Anti-pattern detection
    
    # SELECT *
    if re.search(r'SELECT\s+\*', sql_upper):
        issues.append(Issue(
            severity="medium",
            pattern="SELECT *",
            description="Selecting all columns increases I/O and prevents covering index usage",
            suggestion="Specify only needed columns: SELECT col1, col2, ..."
        ))
    
    # Function on indexed column in WHERE
    func_in_where = re.search(r'WHERE\s+.*?(UPPER|LOWER|TRIM|YEAR|MONTH|DATE|CAST|CONVERT)\s*\(', sql_upper)
    if func_in_where:
        issues.append(Issue(
            severity="high",
            pattern="Function in WHERE clause",
            description=f"Function {func_in_where.group(1)}() prevents index usage on the column",
            suggestion="Create functional index or rewrite condition"
        ))
    
    # Leading wildcard LIKE
    if re.search(r"LIKE\s+['\"]%", sql_upper):
        issues.append(Issue(
            severity="high",
            pattern="Leading wildcard LIKE",
            description="LIKE '%pattern' cannot use indexes, causes full scan",
            suggestion="Use full-text search or reverse index pattern"
        ))
    
    # Large OFFSET
    offset_match = re.search(r'OFFSET\s+(\d+)', sql_upper)
    if offset_match and int(offset_match.group(1)) > 1000:
        issues.append(Issue(
            severity="high",
            pattern="Large OFFSET",
            description=f"OFFSET {offset_match.group(1)} requires scanning and discarding rows",
            suggestion="Use keyset/cursor pagination with WHERE id > last_id"
        ))
    
    # NOT IN (potential NULL issue)
    if re.search(r'NOT\s+IN\s*\(\s*SELECT', sql_upper):
        issues.append(Issue(
            severity="medium",
            pattern="NOT IN with subquery",
            description="NOT IN returns no rows if subquery contains NULL",
            suggestion="Use NOT EXISTS instead for correct NULL handling"
        ))
    
    # OR across different columns
    or_match = re.search(r'WHERE.*?(\w+)\s*=.*?\bOR\b.*?(\w+)\s*=', sql_upper)
    if or_match and or_match.group(1) != or_match.group(2):
        issues.append(Issue(
            severity="medium",
            pattern="OR across columns",
            description="OR conditions on different columns often prevent index usage",
            suggestion="Consider UNION ALL or separate queries"
        ))
    
    # Cartesian product risk
    if metrics["joins"] > 0 and 'ON' not in sql_upper and 'USING' not in sql_upper:
        issues.append(Issue(
            severity="high",
            pattern="Possible Cartesian product",
            description="JOIN without ON/USING clause creates cross product",
            suggestion="Add explicit JOIN conditions"
        ))
    
    # DISTINCT with large result set indicators
    if 'DISTINCT' in sql_upper and 'LIMIT' not in sql_upper:
        issues.append(Issue(
            severity="low",
            pattern="DISTINCT without LIMIT",
            description="DISTINCT requires sorting/hashing all rows",
            suggestion="Verify DISTINCT is necessary; consider GROUP BY or EXISTS"
        ))
    
    # ORDER BY without LIMIT
    if 'ORDER BY' in sql_upper and 'LIMIT' not in sql_upper and metrics["subqueries"] == 0:
        issues.append(Issue(
            severity="low",
            pattern="ORDER BY without LIMIT",
            description="Sorting entire result set may be unnecessary",
            suggestion="Add LIMIT if only top N rows needed"
        ))

    # Large IN list
    in_match = re.search(r'IN\s*\((.*?)\)', sql, re.IGNORECASE | re.DOTALL)
    if in_match:
        items = in_match.group(1).count(',') + 1
        if items > 100:
            issues.append(Issue(
                severity="medium",
                pattern=f"Large IN list ({items} items)",
                description="Large IN lists may degrade to full scan and bloat query plans",
                suggestion="Use temporary table + JOIN or batch into multiple queries"
            ))

    # Missing table alias with JOINs
    if metrics["joins"] > 0 and not re.search(r'\b\w+\.\w+\b', sql):
        issues.append(Issue(
            severity="low",
            pattern="No table aliases with JOINs",
            description="Ambiguous column references can cause errors or wrong index selection",
            suggestion="Use table aliases: SELECT t.col FROM table t JOIN ..."
        ))

    # Calculate complexity score (1-10)
    complexity = 1
    complexity += min(metrics["joins"] * 1.5, 3)
    complexity += min(metrics["subqueries"] * 2, 3)
    complexity += min(metrics["conditions"] * 0.3, 1.5)
    complexity += min(metrics["aggregations"] * 0.5, 1.5)
    complexity += len([i for i in issues if i.severity == "high"]) * 0.5
    complexity_score = min(int(complexity), 10)
    
    return AnalysisResult(
        complexity_score=complexity_score,
        issues=[asdict(i) for i in issues],
        metrics=metrics
    )

def main():
    parser = argparse.ArgumentParser(description='Analyze SQL query complexity')
    parser.add_argument('query', nargs='?', help='SQL query string')
    parser.add_argument('--file', '-f', help='Read query from file')
    parser.add_argument('--json', '-j', action='store_true', help='Output as JSON')
    args = parser.parse_args()
    
    if args.file:
        try:
            with open(args.file) as f:
                sql = f.read()
        except FileNotFoundError:
            print(f"Error: File '{args.file}' not found", file=sys.stderr)
            sys.exit(1)
    elif args.query:
        sql = args.query
    else:
        # Read from stdin
        sql = sys.stdin.read()
    
    if not sql.strip():
        print("Error: No query provided", file=sys.stderr)
        print("Usage: python analyze_query.py \"SELECT ...\"", file=sys.stderr)
        print("       python analyze_query.py --file query.sql", file=sys.stderr)
        sys.exit(1)

    # Validate SQL looks like a query
    sql_upper = sql.upper().strip()
    if not any(sql_upper.startswith(kw) for kw in ['SELECT', 'INSERT', 'UPDATE', 'DELETE', 'WITH']):
        print("Warning: Input doesn't look like SQL. Proceeding anyway...", file=sys.stderr)

    result = analyze_query(sql)
    
    if args.json:
        print(json.dumps(asdict(result), indent=2))
    else:
        print(f"Complexity Score: {result.complexity_score}/10")
        print(f"\nMetrics:")
        for k, v in result.metrics.items():
            print(f"  {k}: {v}")
        
        if result.issues:
            print(f"\nIssues Found: {len(result.issues)}")
            for i, issue in enumerate(result.issues, 1):
                print(f"\n  [{issue['severity'].upper()}] {issue['pattern']}")
                print(f"    Problem: {issue['description']}")
                print(f"    Fix: {issue['suggestion']}")
        else:
            print("\nNo issues detected.")

if __name__ == "__main__":
    main()
