#!/usr/bin/env python3
"""
Analyze database schema for optimization opportunities.
Usage: python analyze_schema.py schema.sql
       python analyze_schema.py --file schema.sql --json
"""

import re
import sys
import json
import argparse
from dataclasses import dataclass, asdict
from typing import List, Dict, Optional, Set

@dataclass
class TableInfo:
    name: str
    columns: List[Dict]
    primary_key: Optional[List[str]]
    indexes: List[Dict]
    foreign_keys: List[Dict]
    row_estimate: Optional[int]

@dataclass
class SchemaIssue:
    severity: str
    table: str
    issue: str
    suggestion: str

@dataclass
class SchemaAnalysis:
    tables: List[TableInfo]
    issues: List[SchemaIssue]
    index_suggestions: List[Dict]
    summary: str

def parse_create_table(sql: str) -> List[TableInfo]:
    """Parse CREATE TABLE statements."""
    tables = []
    
    # Find all CREATE TABLE statements
    pattern = r'CREATE\s+TABLE\s+(?:IF\s+NOT\s+EXISTS\s+)?[`"\[]?(\w+)[`"\]]?\s*\((.*?)\)(?:\s*ENGINE|;|\s*$)'
    matches = re.finditer(pattern, sql, re.IGNORECASE | re.DOTALL)
    
    for match in matches:
        table_name = match.group(1).lower()
        body = match.group(2)
        
        columns = []
        primary_key = None
        indexes = []
        foreign_keys = []
        
        # Split by comma, handling nested parentheses
        parts = []
        depth = 0
        current = ""
        for char in body:
            if char == '(':
                depth += 1
            elif char == ')':
                depth -= 1
            elif char == ',' and depth == 0:
                parts.append(current.strip())
                current = ""
                continue
            current += char
        if current.strip():
            parts.append(current.strip())
        
        for part in parts:
            part_upper = part.upper().strip()
            
            # Primary key constraint
            if part_upper.startswith('PRIMARY KEY'):
                pk_match = re.search(r'\(([^)]+)\)', part)
                if pk_match:
                    primary_key = [c.strip().strip('`"[]') for c in pk_match.group(1).split(',')]
            
            # Index/Key
            elif re.match(r'(UNIQUE\s+)?(KEY|INDEX)', part_upper):
                idx_match = re.search(r'(?:KEY|INDEX)\s+[`"\[]?(\w+)[`"\]]?\s*\(([^)]+)\)', part, re.IGNORECASE)
                if idx_match:
                    idx_name = idx_match.group(1)
                    idx_cols = [c.strip().strip('`"[]').split()[0] for c in idx_match.group(2).split(',')]
                    indexes.append({
                        'name': idx_name,
                        'columns': idx_cols,
                        'unique': 'UNIQUE' in part_upper
                    })
            
            # Foreign key
            elif part_upper.startswith('FOREIGN KEY') or 'REFERENCES' in part_upper:
                fk_match = re.search(r'FOREIGN\s+KEY\s*\(([^)]+)\)\s*REFERENCES\s+[`"\[]?(\w+)[`"\]]?\s*\(([^)]+)\)', part, re.IGNORECASE)
                if fk_match:
                    foreign_keys.append({
                        'columns': [c.strip().strip('`"[]') for c in fk_match.group(1).split(',')],
                        'references_table': fk_match.group(2),
                        'references_columns': [c.strip().strip('`"[]') for c in fk_match.group(3).split(',')]
                    })
            
            # Column definition
            elif not part_upper.startswith(('CONSTRAINT', 'CHECK', 'UNIQUE')):
                col_match = re.match(r'[`"\[]?(\w+)[`"\]]?\s+(\w+(?:\([^)]+\))?)', part, re.IGNORECASE)
                if col_match:
                    col_name = col_match.group(1).lower()
                    col_type = col_match.group(2).upper()
                    
                    # Check for inline primary key
                    if 'PRIMARY KEY' in part_upper:
                        primary_key = [col_name]
                    
                    columns.append({
                        'name': col_name,
                        'type': col_type,
                        'nullable': 'NOT NULL' not in part_upper,
                        'has_default': 'DEFAULT' in part_upper,
                        'auto_increment': 'AUTO_INCREMENT' in part_upper or 'SERIAL' in part_upper
                    })
        
        tables.append(TableInfo(
            name=table_name,
            columns=columns,
            primary_key=primary_key,
            indexes=indexes,
            foreign_keys=foreign_keys,
            row_estimate=None
        ))
    
    return tables

def analyze_schema(tables: List[TableInfo]) -> SchemaAnalysis:
    """Analyze schema for issues and suggestions."""
    issues = []
    index_suggestions = []
    
    # Build FK reference map
    fk_targets: Dict[str, Set[str]] = {}  # table -> set of columns referenced by FKs
    for table in tables:
        for fk in table.foreign_keys:
            ref_table = fk['references_table'].lower()
            if ref_table not in fk_targets:
                fk_targets[ref_table] = set()
            fk_targets[ref_table].update(c.lower() for c in fk['references_columns'])
    
    for table in tables:
        indexed_columns: Set[str] = set()
        indexed_prefixes: List[List[str]] = []
        
        # Collect indexed columns
        if table.primary_key:
            indexed_columns.update(c.lower() for c in table.primary_key)
            indexed_prefixes.append([c.lower() for c in table.primary_key])
        
        for idx in table.indexes:
            indexed_columns.update(c.lower() for c in idx['columns'])
            indexed_prefixes.append([c.lower() for c in idx['columns']])
        
        # Check 1: Missing primary key
        if not table.primary_key:
            issues.append(SchemaIssue(
                severity="high",
                table=table.name,
                issue="Table has no primary key",
                suggestion="Add a primary key (surrogate or natural) for row identification"
            ))
        
        # Check 2: Foreign keys without indexes
        for fk in table.foreign_keys:
            fk_cols = [c.lower() for c in fk['columns']]
            has_index = any(
                prefix[:len(fk_cols)] == fk_cols 
                for prefix in indexed_prefixes
            )
            if not has_index:
                issues.append(SchemaIssue(
                    severity="high",
                    table=table.name,
                    issue=f"Foreign key columns ({', '.join(fk_cols)}) not indexed",
                    suggestion=f"Add index for JOIN performance on {table.name}"
                ))
                index_suggestions.append({
                    'table': table.name,
                    'columns': fk_cols,
                    'reason': 'foreign_key',
                    'sql': f"CREATE INDEX idx_{table.name}_{'_'.join(fk_cols)} ON {table.name}({', '.join(fk_cols)});"
                })
        
        # Check 3: Common column patterns without indexes
        common_filter_cols = ['status', 'type', 'state', 'active', 'deleted', 'is_active', 'is_deleted']
        common_sort_cols = ['created_at', 'updated_at', 'created', 'modified', 'timestamp', 'date']
        
        for col in table.columns:
            col_name = col['name'].lower()
            
            # Status-like columns often need composite indexes
            if col_name in common_filter_cols and col_name not in indexed_columns:
                issues.append(SchemaIssue(
                    severity="low",
                    table=table.name,
                    issue=f"Column '{col_name}' (filter candidate) is not indexed",
                    suggestion="Consider composite index with high-cardinality column"
                ))
            
            # Timestamp columns for sorting
            if col_name in common_sort_cols and col_name not in indexed_columns:
                issues.append(SchemaIssue(
                    severity="medium",
                    table=table.name,
                    issue=f"Timestamp column '{col_name}' is not indexed",
                    suggestion="Add index for ORDER BY queries; consider BRIN for time-series"
                ))
                index_suggestions.append({
                    'table': table.name,
                    'columns': [col_name],
                    'reason': 'timestamp_sort',
                    'sql': f"CREATE INDEX idx_{table.name}_{col_name} ON {table.name}({col_name});"
                })
        
        # Check 4: Wide tables without covering indexes hint
        if len(table.columns) > 20:
            issues.append(SchemaIssue(
                severity="low",
                table=table.name,
                issue=f"Wide table ({len(table.columns)} columns)",
                suggestion="Consider covering indexes for frequent queries to avoid heap fetches"
            ))
        
        # Check 5: Text columns that might need full-text
        for col in table.columns:
            if col['type'] in ('TEXT', 'LONGTEXT', 'MEDIUMTEXT', 'VARCHAR(MAX)'):
                issues.append(SchemaIssue(
                    severity="low",
                    table=table.name,
                    issue=f"Large text column '{col['name']}' may need special handling",
                    suggestion="Consider full-text index if searched; exclude from SELECT * queries"
                ))
        
        # Check 6: Redundant indexes
        for i, idx1 in enumerate(table.indexes):
            for idx2 in table.indexes[i+1:]:
                cols1 = [c.lower() for c in idx1['columns']]
                cols2 = [c.lower() for c in idx2['columns']]
                
                # Check if one is prefix of another
                min_len = min(len(cols1), len(cols2))
                if cols1[:min_len] == cols2[:min_len]:
                    shorter = idx1['name'] if len(cols1) < len(cols2) else idx2['name']
                    longer = idx2['name'] if len(cols1) < len(cols2) else idx1['name']
                    issues.append(SchemaIssue(
                        severity="medium",
                        table=table.name,
                        issue=f"Potentially redundant indexes: {shorter} is prefix of {longer}",
                        suggestion=f"Consider dropping {shorter} if queries use {longer}"
                    ))
    
    # Generate summary
    high_count = sum(1 for i in issues if i.severity == "high")
    med_count = sum(1 for i in issues if i.severity == "medium")
    
    if high_count > 0:
        summary = f"Found {high_count} high-priority issues requiring attention."
    elif med_count > 0:
        summary = f"Schema looks reasonable with {med_count} optimization opportunities."
    else:
        summary = "Schema follows good practices."
    
    return SchemaAnalysis(
        tables=[asdict(t) for t in tables],
        issues=[asdict(i) for i in issues],
        index_suggestions=index_suggestions,
        summary=summary
    )

def main():
    parser = argparse.ArgumentParser(description='Analyze database schema')
    parser.add_argument('file', nargs='?', help='SQL file with CREATE TABLE statements')
    parser.add_argument('--json', '-j', action='store_true', help='Output as JSON')
    args = parser.parse_args()
    
    if args.file:
        try:
            with open(args.file) as f:
                sql = f.read()
        except FileNotFoundError:
            print(f"Error: File '{args.file}' not found", file=sys.stderr)
            sys.exit(1)
    else:
        sql = sys.stdin.read()
    
    if not sql.strip():
        print("Error: No SQL provided", file=sys.stderr)
        sys.exit(1)
    
    tables = parse_create_table(sql)
    
    if not tables:
        print("Error: No CREATE TABLE statements found", file=sys.stderr)
        sys.exit(1)
    
    result = analyze_schema(tables)
    
    if args.json:
        print(json.dumps(asdict(result), indent=2))
    else:
        print(f"Tables analyzed: {len(result.tables)}")
        print(f"Summary: {result.summary}\n")
        
        for t in result.tables:
            print(f"📋 {t['name']}")
            print(f"   Columns: {len(t['columns'])}")
            print(f"   Primary Key: {', '.join(t['primary_key']) if t['primary_key'] else 'None'}")
            print(f"   Indexes: {len(t['indexes'])}")
            print(f"   Foreign Keys: {len(t['foreign_keys'])}")
            print()
        
        if result.issues:
            print(f"Issues Found: {len(result.issues)}\n")
            for issue in result.issues:
                print(f"  [{issue['severity'].upper()}] {issue['table']}: {issue['issue']}")
                print(f"    → {issue['suggestion']}")
                print()
        
        if result.index_suggestions:
            print("Suggested Indexes:")
            for idx in result.index_suggestions:
                print(f"  {idx['sql']}")

if __name__ == "__main__":
    main()
