#!/usr/bin/env python3
"""
Check database migrations for potential safety issues.
Detects operations that may lock tables or cause downtime.

Usage: python check_migration.py migration.sql
       python check_migration.py --pg migration.sql
       python check_migration.py --mysql migration.sql --json
"""

import re
import sys
import json
import argparse
from dataclasses import dataclass, asdict
from typing import List, Optional

@dataclass
class MigrationIssue:
    severity: str  # "critical", "warning", "info"
    line: int
    operation: str
    statement: str
    problem: str
    safe_alternative: str

@dataclass
class MigrationAnalysis:
    database: str
    total_statements: int
    issues: List[MigrationIssue]
    estimated_risk: str  # "high", "medium", "low"
    summary: str

def analyze_postgresql(sql: str) -> List[MigrationIssue]:
    """Analyze PostgreSQL migration for safety issues."""
    issues = []
    statements = re.split(r';\s*\n', sql)
    
    line_num = 1
    for stmt in statements:
        stmt_upper = stmt.upper().strip()
        stmt_clean = stmt.strip()
        
        if not stmt_clean:
            line_num += stmt.count('\n') + 1
            continue
        
        # CREATE INDEX without CONCURRENTLY
        if re.search(r'CREATE\s+(?:UNIQUE\s+)?INDEX\s+(?!CONCURRENTLY)', stmt_upper):
            issues.append(MigrationIssue(
                severity="critical",
                line=line_num,
                operation="CREATE INDEX",
                statement=stmt_clean[:100],
                problem="Blocks writes (INSERT, UPDATE, DELETE) on the table until complete",
                safe_alternative="Use CREATE INDEX CONCURRENTLY (requires separate transaction)"
            ))
        
        # DROP INDEX without CONCURRENTLY
        if re.search(r'DROP\s+INDEX\s+(?!CONCURRENTLY)', stmt_upper):
            issues.append(MigrationIssue(
                severity="warning",
                line=line_num,
                operation="DROP INDEX",
                statement=stmt_clean[:100],
                problem="Briefly blocks reads and writes",
                safe_alternative="Use DROP INDEX CONCURRENTLY"
            ))
        
        # ALTER TABLE ADD COLUMN with DEFAULT (pre-PG11 issue)
        if re.search(r'ALTER\s+TABLE.*ADD\s+(?:COLUMN\s+)?.*DEFAULT', stmt_upper):
            issues.append(MigrationIssue(
                severity="info",
                line=line_num,
                operation="ADD COLUMN WITH DEFAULT",
                statement=stmt_clean[:100],
                problem="In PostgreSQL <11, rewrites entire table. PG11+ is fast.",
                safe_alternative="For PG<11: Add nullable column, set default, then add NOT NULL"
            ))
        
        # ALTER TABLE ADD COLUMN NOT NULL without DEFAULT
        if re.search(r'ALTER\s+TABLE.*ADD\s+(?:COLUMN\s+)?\w+.*NOT\s+NULL(?!\s+DEFAULT)', stmt_upper):
            issues.append(MigrationIssue(
                severity="critical",
                line=line_num,
                operation="ADD NOT NULL COLUMN",
                statement=stmt_clean[:100],
                problem="Fails if table has existing rows (no default value)",
                safe_alternative="Add with DEFAULT, or make nullable first and backfill"
            ))
        
        # ALTER TABLE SET NOT NULL
        if re.search(r'ALTER\s+TABLE.*ALTER\s+COLUMN.*SET\s+NOT\s+NULL', stmt_upper):
            issues.append(MigrationIssue(
                severity="warning",
                line=line_num,
                operation="SET NOT NULL",
                statement=stmt_clean[:100],
                problem="Full table scan to verify no NULLs exist; blocks writes",
                safe_alternative="Add CHECK constraint with NOT VALID, validate separately, then convert"
            ))
        
        # ALTER TABLE SET TYPE (column type change)
        if re.search(r'ALTER\s+TABLE.*ALTER\s+COLUMN.*(?:SET\s+DATA\s+)?TYPE', stmt_upper):
            issues.append(MigrationIssue(
                severity="critical",
                line=line_num,
                operation="CHANGE COLUMN TYPE",
                statement=stmt_clean[:100],
                problem="Rewrites entire table; exclusive lock until complete",
                safe_alternative="Create new column, backfill data, switch (3-step migration)"
            ))
        
        # RENAME TABLE/COLUMN
        if re.search(r'ALTER\s+TABLE.*RENAME', stmt_upper):
            issues.append(MigrationIssue(
                severity="warning",
                line=line_num,
                operation="RENAME",
                statement=stmt_clean[:100],
                problem="Brief exclusive lock; application code must update simultaneously",
                safe_alternative="Consider dual-write pattern during transition"
            ))
        
        # ADD FOREIGN KEY
        if re.search(r'ADD\s+(?:CONSTRAINT\s+\w+\s+)?FOREIGN\s+KEY', stmt_upper):
            issues.append(MigrationIssue(
                severity="warning",
                line=line_num,
                operation="ADD FOREIGN KEY",
                statement=stmt_clean[:100],
                problem="Validates all existing rows; blocks writes during scan",
                safe_alternative="Use NOT VALID initially, then VALIDATE CONSTRAINT separately"
            ))
        
        # ADD CHECK CONSTRAINT (validated)
        if re.search(r'ADD\s+(?:CONSTRAINT\s+\w+\s+)?CHECK\s+', stmt_upper) and 'NOT VALID' not in stmt_upper:
            issues.append(MigrationIssue(
                severity="warning",
                line=line_num,
                operation="ADD CHECK CONSTRAINT",
                statement=stmt_clean[:100],
                problem="Scans entire table to validate; holds lock",
                safe_alternative="Add with NOT VALID, then VALIDATE CONSTRAINT in separate transaction"
            ))
        
        # LOCK TABLE
        if re.search(r'LOCK\s+TABLE', stmt_upper):
            issues.append(MigrationIssue(
                severity="critical",
                line=line_num,
                operation="LOCK TABLE",
                statement=stmt_clean[:100],
                problem="Explicit table lock blocks other transactions",
                safe_alternative="Avoid if possible; use advisory locks for coordination"
            ))
        
        # VACUUM FULL
        if re.search(r'VACUUM\s+FULL', stmt_upper):
            issues.append(MigrationIssue(
                severity="critical",
                line=line_num,
                operation="VACUUM FULL",
                statement=stmt_clean[:100],
                problem="Exclusive lock; rewrites entire table; long downtime",
                safe_alternative="Use pg_repack for online table rewrite"
            ))
        
        # CLUSTER
        if re.search(r'\bCLUSTER\b', stmt_upper):
            issues.append(MigrationIssue(
                severity="critical",
                line=line_num,
                operation="CLUSTER",
                statement=stmt_clean[:100],
                problem="Exclusive lock; rewrites entire table",
                safe_alternative="Use pg_repack or schedule during maintenance window"
            ))
        
        # REINDEX without CONCURRENTLY
        if re.search(r'REINDEX\s+(?!.*CONCURRENTLY)', stmt_upper):
            issues.append(MigrationIssue(
                severity="warning",
                line=line_num,
                operation="REINDEX",
                statement=stmt_clean[:100],
                problem="Blocks reads and writes on the table",
                safe_alternative="Use REINDEX CONCURRENTLY (PostgreSQL 12+)"
            ))
        
        line_num += stmt.count('\n') + 1
    
    return issues

def analyze_mysql(sql: str) -> List[MigrationIssue]:
    """Analyze MySQL migration for safety issues."""
    issues = []
    statements = re.split(r';\s*\n', sql)
    
    line_num = 1
    for stmt in statements:
        stmt_upper = stmt.upper().strip()
        stmt_clean = stmt.strip()
        
        if not stmt_clean:
            line_num += stmt.count('\n') + 1
            continue
        
        # ALTER TABLE (most ALTERs in MySQL are problematic)
        if re.search(r'ALTER\s+TABLE', stmt_upper):
            # ADD INDEX / ADD KEY
            if re.search(r'ADD\s+(?:UNIQUE\s+)?(?:INDEX|KEY)', stmt_upper):
                if 'ALGORITHM=INPLACE' not in stmt_upper and 'LOCK=NONE' not in stmt_upper:
                    issues.append(MigrationIssue(
                        severity="warning",
                        line=line_num,
                        operation="ADD INDEX",
                        statement=stmt_clean[:100],
                        problem="May block writes depending on MySQL version and storage engine",
                        safe_alternative="Use ALGORITHM=INPLACE, LOCK=NONE (MySQL 5.6+)"
                    ))
            
            # ADD COLUMN
            elif re.search(r'ADD\s+(?:COLUMN\s+)?', stmt_upper):
                if 'ALGORITHM=INSTANT' not in stmt_upper:
                    issues.append(MigrationIssue(
                        severity="warning",
                        line=line_num,
                        operation="ADD COLUMN",
                        statement=stmt_clean[:100],
                        problem="May rebuild table in older MySQL versions",
                        safe_alternative="Use ALGORITHM=INSTANT (MySQL 8.0+) for adding columns at end"
                    ))
            
            # MODIFY/CHANGE COLUMN
            if re.search(r'(?:MODIFY|CHANGE)\s+(?:COLUMN\s+)?', stmt_upper):
                issues.append(MigrationIssue(
                    severity="critical",
                    line=line_num,
                    operation="MODIFY COLUMN",
                    statement=stmt_clean[:100],
                    problem="Usually rebuilds entire table; long lock",
                    safe_alternative="Use pt-online-schema-change or gh-ost for large tables"
                ))
            
            # DROP COLUMN
            if re.search(r'DROP\s+(?:COLUMN\s+)?', stmt_upper):
                issues.append(MigrationIssue(
                    severity="critical",
                    line=line_num,
                    operation="DROP COLUMN",
                    statement=stmt_clean[:100],
                    problem="Rebuilds entire table; exclusive lock",
                    safe_alternative="Use pt-online-schema-change or gh-ost"
                ))
            
            # ADD FOREIGN KEY
            if re.search(r'ADD\s+(?:CONSTRAINT\s+\w+\s+)?FOREIGN\s+KEY', stmt_upper):
                issues.append(MigrationIssue(
                    severity="warning",
                    line=line_num,
                    operation="ADD FOREIGN KEY",
                    statement=stmt_clean[:100],
                    problem="Validates existing data; may lock both tables",
                    safe_alternative="Create index first, add FK with SET FOREIGN_KEY_CHECKS=0 (carefully)"
                ))
        
        # RENAME TABLE
        if re.search(r'RENAME\s+TABLE', stmt_upper):
            issues.append(MigrationIssue(
                severity="warning",
                line=line_num,
                operation="RENAME TABLE",
                statement=stmt_clean[:100],
                problem="Atomic but requires brief metadata lock",
                safe_alternative="Coordinate with application deployment"
            ))
        
        # OPTIMIZE TABLE
        if re.search(r'OPTIMIZE\s+TABLE', stmt_upper):
            issues.append(MigrationIssue(
                severity="critical",
                line=line_num,
                operation="OPTIMIZE TABLE",
                statement=stmt_clean[:100],
                problem="Rebuilds and locks table for duration",
                safe_alternative="Use pt-online-schema-change or schedule during maintenance"
            ))
        
        line_num += stmt.count('\n') + 1
    
    return issues

def main():
    parser = argparse.ArgumentParser(description='Check migration safety')
    parser.add_argument('file', nargs='?', help='Migration SQL file')
    parser.add_argument('--pg', '--postgresql', action='store_true', help='PostgreSQL syntax')
    parser.add_argument('--mysql', action='store_true', help='MySQL syntax')
    parser.add_argument('--json', '-j', action='store_true', help='Output as JSON')
    args = parser.parse_args()
    
    if not args.pg and not args.mysql:
        # Auto-detect based on keywords
        pass  # Will analyze for both
    
    if args.file:
        try:
            with open(args.file) as f:
                sql = f.read()
        except FileNotFoundError:
            print(f"Error: File '{args.file}' not found", file=sys.stderr)
            sys.exit(1)
    else:
        sql = sys.stdin.read()
    
    if not sql.strip():
        print("Error: No SQL provided", file=sys.stderr)
        sys.exit(1)
    
    # Analyze
    if args.pg:
        issues = analyze_postgresql(sql)
        db = 'postgresql'
    elif args.mysql:
        issues = analyze_mysql(sql)
        db = 'mysql'
    else:
        # Try both, use one with more findings or default to PostgreSQL patterns
        pg_issues = analyze_postgresql(sql)
        mysql_issues = analyze_mysql(sql)
        if len(mysql_issues) > len(pg_issues):
            issues = mysql_issues
            db = 'mysql'
        else:
            issues = pg_issues
            db = 'postgresql'
    
    # Count statements
    total_statements = len([s for s in re.split(r';\s*\n', sql) if s.strip()])
    
    # Determine risk
    critical = sum(1 for i in issues if i.severity == 'critical')
    warnings = sum(1 for i in issues if i.severity == 'warning')
    
    if critical > 0:
        risk = 'high'
        summary = f"DANGER: {critical} critical issues found that WILL cause downtime on large tables."
    elif warnings > 0:
        risk = 'medium'
        summary = f"CAUTION: {warnings} warnings found that may cause issues on large tables."
    else:
        risk = 'low'
        summary = "Migration appears safe for online deployment."
    
    result = MigrationAnalysis(
        database=db,
        total_statements=total_statements,
        issues=[asdict(i) for i in issues],
        estimated_risk=risk,
        summary=summary
    )
    
    if args.json:
        print(json.dumps(asdict(result), indent=2))
    else:
        print(f"Database: {result.database.upper()}")
        print(f"Statements: {result.total_statements}")
        print(f"Risk Level: {result.estimated_risk.upper()}")
        print(f"\n{result.summary}\n")
        
        if issues:
            print(f"{'='*60}")
            print(f"Issues Found: {len(issues)}")
            print(f"{'='*60}\n")
            
            for issue in issues:
                icon = "🔴" if issue.severity == 'critical' else "🟡" if issue.severity == 'warning' else "🔵"
                print(f"{icon} Line {issue.line}: {issue.operation}")
                print(f"   Statement: {issue.statement}...")
                print(f"   Problem: {issue.problem}")
                print(f"   Safe Alternative: {issue.safe_alternative}")
                print()
        else:
            print("✅ No safety issues detected.")

if __name__ == "__main__":
    main()
