#!/usr/bin/env python3
"""
Compare two EXPLAIN outputs to verify optimization impact.
Shows improvements/regressions in scan types, rows, timing, and buffers.

Usage: 
  python diff_explain.py --pg before.txt after.txt
  python diff_explain.py --mysql before.txt after.txt
  python diff_explain.py --pg --before "$(cat before.txt)" --after "$(cat after.txt)"
  python diff_explain.py --pg before.txt after.txt --json
"""

import re
import sys
import json
import argparse
from dataclasses import dataclass, asdict, field
from typing import List, Dict, Optional, Tuple
from enum import Enum

class ChangeType(Enum):
    IMPROVED = "improved"
    REGRESSED = "regressed"
    UNCHANGED = "unchanged"
    ADDED = "added"
    REMOVED = "removed"

@dataclass
class PlanMetrics:
    """Extracted metrics from an EXPLAIN output."""
    scan_types: List[str] = field(default_factory=list)
    total_cost: Optional[float] = None
    estimated_rows: Optional[int] = None
    actual_rows: Optional[int] = None
    execution_time_ms: Optional[float] = None
    planning_time_ms: Optional[float] = None
    shared_buffers_hit: Optional[int] = None
    shared_buffers_read: Optional[int] = None
    temp_buffers: Optional[int] = None
    sort_method: Optional[str] = None
    sort_space_kb: Optional[int] = None
    join_types: List[str] = field(default_factory=list)
    index_names: List[str] = field(default_factory=list)
    tables_scanned: List[str] = field(default_factory=list)
    warnings: List[str] = field(default_factory=list)

@dataclass
class MetricComparison:
    """Comparison of a single metric."""
    metric: str
    before: Optional[str]
    after: Optional[str]
    change_type: str
    change_pct: Optional[float] = None
    impact: str = ""

@dataclass 
class DiffResult:
    """Complete diff result."""
    database: str
    summary: str
    overall_verdict: str  # "improved", "regressed", "mixed", "unchanged"
    comparisons: List[MetricComparison]
    scan_type_changes: Dict[str, str]
    recommendations: List[str]

def parse_postgresql_explain(output: str) -> PlanMetrics:
    """Parse PostgreSQL EXPLAIN (ANALYZE, BUFFERS) output."""
    metrics = PlanMetrics()
    
    # Scan types
    scan_patterns = [
        (r'Seq Scan on (\w+)', 'Seq Scan'),
        (r'Index Scan using (\w+) on (\w+)', 'Index Scan'),
        (r'Index Only Scan using (\w+) on (\w+)', 'Index Only Scan'),
        (r'Bitmap Heap Scan on (\w+)', 'Bitmap Heap Scan'),
        (r'Bitmap Index Scan on (\w+)', 'Bitmap Index Scan'),
    ]
    
    for pattern, scan_type in scan_patterns:
        for match in re.finditer(pattern, output, re.IGNORECASE):
            metrics.scan_types.append(scan_type)
            if 'Index' in scan_type:
                # Extract index name
                idx_name = match.group(1)
                if idx_name not in metrics.index_names:
                    metrics.index_names.append(idx_name)
            # Extract table name
            table = match.group(2) if match.lastindex >= 2 else match.group(1)
            if table not in metrics.tables_scanned:
                metrics.tables_scanned.append(table)
    
    # Join types
    join_patterns = ['Nested Loop', 'Hash Join', 'Merge Join']
    for join_type in join_patterns:
        if join_type in output:
            metrics.join_types.append(join_type)
    
    # Cost
    cost_match = re.search(r'cost=[\d.]+\.\.([\d.]+)', output)
    if cost_match:
        metrics.total_cost = float(cost_match.group(1))
    
    # Rows (first occurrence is usually the top-level)
    rows_match = re.search(r'rows=(\d+).*?(?:actual.*?rows=(\d+))?', output, re.IGNORECASE)
    if rows_match:
        metrics.estimated_rows = int(rows_match.group(1))
        if rows_match.group(2):
            metrics.actual_rows = int(rows_match.group(2))
    
    # Execution time
    exec_match = re.search(r'Execution Time:\s*([\d.]+)\s*ms', output, re.IGNORECASE)
    if exec_match:
        metrics.execution_time_ms = float(exec_match.group(1))
    
    # Planning time
    plan_match = re.search(r'Planning Time:\s*([\d.]+)\s*ms', output, re.IGNORECASE)
    if plan_match:
        metrics.planning_time_ms = float(plan_match.group(1))
    
    # Buffer usage
    hit_match = re.search(r'Buffers:.*?shared hit=(\d+)', output, re.IGNORECASE)
    if hit_match:
        metrics.shared_buffers_hit = int(hit_match.group(1))
    
    read_match = re.search(r'Buffers:.*?(?:shared )?read=(\d+)', output, re.IGNORECASE)
    if read_match:
        metrics.shared_buffers_read = int(read_match.group(1))
    
    temp_match = re.search(r'Buffers:.*?temp.*?=(\d+)', output, re.IGNORECASE)
    if temp_match:
        metrics.temp_buffers = int(temp_match.group(1))
    
    # Sort info
    sort_match = re.search(r'Sort Method:\s*(\w+(?:\s+\w+)?)', output, re.IGNORECASE)
    if sort_match:
        metrics.sort_method = sort_match.group(1)
    
    sort_space = re.search(r'Sort.*?Memory:\s*(\d+)kB', output, re.IGNORECASE)
    if sort_space:
        metrics.sort_space_kb = int(sort_space.group(1))
    
    # Warnings
    if 'external' in output.lower() and 'sort' in output.lower():
        metrics.warnings.append('Sort spilled to disk')
    if re.search(r'rows=\d+.*?actual.*?rows=(\d+)', output) and metrics.estimated_rows and metrics.actual_rows:
        ratio = metrics.actual_rows / max(metrics.estimated_rows, 1)
        if ratio > 10 or ratio < 0.1:
            metrics.warnings.append(f'Row estimate off by {ratio:.1f}x')
    
    return metrics

def parse_mysql_explain(output: str) -> PlanMetrics:
    """Parse MySQL EXPLAIN output."""
    metrics = PlanMetrics()
    
    # Detect type column values
    type_map = {
        'all': 'Full Table Scan',
        'index': 'Full Index Scan', 
        'range': 'Index Range Scan',
        'ref': 'Index Lookup (ref)',
        'eq_ref': 'Index Lookup (eq_ref)',
        'const': 'Const Lookup',
        'system': 'System Const',
    }
    
    for mysql_type, scan_type in type_map.items():
        # Match in tabular or JSON format
        if re.search(rf'\b{mysql_type}\b', output, re.IGNORECASE):
            if mysql_type in ('all', 'index', 'range', 'ref', 'eq_ref', 'const'):
                metrics.scan_types.append(scan_type)
    
    # Extract table names
    table_matches = re.finditer(r'table["\s:]+["\']?(\w+)["\']?', output, re.IGNORECASE)
    for match in table_matches:
        table = match.group(1)
        if table not in ('table',) and table not in metrics.tables_scanned:
            metrics.tables_scanned.append(table)
    
    # Index names (key column)
    key_matches = re.finditer(r'(?:key|possible_keys)["\s:]+["\']?(\w+)["\']?', output, re.IGNORECASE)
    for match in key_matches:
        idx = match.group(1)
        if idx.upper() not in ('NULL', 'KEY', 'POSSIBLE_KEYS') and idx not in metrics.index_names:
            metrics.index_names.append(idx)
    
    # Rows examined
    rows_match = re.search(r'rows["\s:]+(\d+)', output, re.IGNORECASE)
    if rows_match:
        metrics.estimated_rows = int(rows_match.group(1))
    
    # Actual time (EXPLAIN ANALYZE format)
    time_match = re.search(r'actual time=([\d.]+)\.\.([\d.]+)', output)
    if time_match:
        metrics.execution_time_ms = float(time_match.group(2))
    
    # Extra column warnings
    extra_lower = output.lower()
    if 'using filesort' in extra_lower:
        metrics.warnings.append('Using filesort')
        metrics.sort_method = 'filesort'
    if 'using temporary' in extra_lower:
        metrics.warnings.append('Using temporary table')
    if 'using where' in extra_lower and 'using index' not in extra_lower:
        metrics.warnings.append('Post-index filtering (Using where)')
    
    return metrics

def compare_metrics(before: PlanMetrics, after: PlanMetrics) -> List[MetricComparison]:
    """Compare two sets of metrics."""
    comparisons = []
    
    # Execution time (lower is better)
    if before.execution_time_ms is not None and after.execution_time_ms is not None:
        change_pct = ((after.execution_time_ms - before.execution_time_ms) / before.execution_time_ms) * 100
        if change_pct < -10:
            change_type = ChangeType.IMPROVED.value
            impact = f"🟢 {abs(change_pct):.1f}% faster"
        elif change_pct > 10:
            change_type = ChangeType.REGRESSED.value
            impact = f"🔴 {change_pct:.1f}% slower"
        else:
            change_type = ChangeType.UNCHANGED.value
            impact = "No significant change"
        
        comparisons.append(MetricComparison(
            metric="Execution Time",
            before=f"{before.execution_time_ms:.2f} ms",
            after=f"{after.execution_time_ms:.2f} ms",
            change_type=change_type,
            change_pct=change_pct,
            impact=impact
        ))
    
    # Total cost (lower is better)
    if before.total_cost is not None and after.total_cost is not None:
        change_pct = ((after.total_cost - before.total_cost) / before.total_cost) * 100
        if change_pct < -10:
            change_type = ChangeType.IMPROVED.value
            impact = f"🟢 {abs(change_pct):.1f}% lower cost"
        elif change_pct > 10:
            change_type = ChangeType.REGRESSED.value
            impact = f"🔴 {change_pct:.1f}% higher cost"
        else:
            change_type = ChangeType.UNCHANGED.value
            impact = "No significant change"
        
        comparisons.append(MetricComparison(
            metric="Estimated Cost",
            before=f"{before.total_cost:.2f}",
            after=f"{after.total_cost:.2f}",
            change_type=change_type,
            change_pct=change_pct,
            impact=impact
        ))
    
    # Buffer reads (lower is better)
    if before.shared_buffers_read is not None and after.shared_buffers_read is not None:
        before_val = before.shared_buffers_read
        after_val = after.shared_buffers_read
        if before_val > 0:
            change_pct = ((after_val - before_val) / before_val) * 100
            if change_pct < -20:
                change_type = ChangeType.IMPROVED.value
                impact = f"🟢 {abs(change_pct):.1f}% fewer disk reads"
            elif change_pct > 20:
                change_type = ChangeType.REGRESSED.value
                impact = f"🔴 {change_pct:.1f}% more disk reads"
            else:
                change_type = ChangeType.UNCHANGED.value
                impact = "Similar disk I/O"
        else:
            change_type = ChangeType.UNCHANGED.value
            change_pct = 0
            impact = "Minimal disk reads"
        
        comparisons.append(MetricComparison(
            metric="Disk Reads (buffers)",
            before=str(before_val),
            after=str(after_val),
            change_type=change_type,
            change_pct=change_pct,
            impact=impact
        ))
    
    # Cache hit ratio (higher is better)
    before_total = (before.shared_buffers_hit or 0) + (before.shared_buffers_read or 0)
    after_total = (after.shared_buffers_hit or 0) + (after.shared_buffers_read or 0)
    
    if before_total > 0 and after_total > 0:
        before_ratio = (before.shared_buffers_hit or 0) / before_total * 100
        after_ratio = (after.shared_buffers_hit or 0) / after_total * 100
        
        if after_ratio > before_ratio + 10:
            change_type = ChangeType.IMPROVED.value
            impact = f"🟢 Better cache utilization"
        elif after_ratio < before_ratio - 10:
            change_type = ChangeType.REGRESSED.value
            impact = f"🔴 Worse cache utilization"
        else:
            change_type = ChangeType.UNCHANGED.value
            impact = "Similar cache usage"
        
        comparisons.append(MetricComparison(
            metric="Cache Hit Ratio",
            before=f"{before_ratio:.1f}%",
            after=f"{after_ratio:.1f}%",
            change_type=change_type,
            change_pct=after_ratio - before_ratio,
            impact=impact
        ))
    
    # Rows examined (comparing actual if available, else estimated)
    before_rows = before.actual_rows or before.estimated_rows
    after_rows = after.actual_rows or after.estimated_rows
    
    if before_rows and after_rows:
        if before_rows > 0:
            change_pct = ((after_rows - before_rows) / before_rows) * 100
            if change_pct < -30:
                change_type = ChangeType.IMPROVED.value
                impact = f"🟢 {abs(change_pct):.1f}% fewer rows examined"
            elif change_pct > 30:
                change_type = ChangeType.REGRESSED.value
                impact = f"🔴 {change_pct:.1f}% more rows examined"
            else:
                change_type = ChangeType.UNCHANGED.value
                impact = "Similar row count"
        else:
            change_type = ChangeType.UNCHANGED.value
            change_pct = 0
            impact = "N/A"
        
        comparisons.append(MetricComparison(
            metric="Rows Examined",
            before=f"{before_rows:,}",
            after=f"{after_rows:,}",
            change_type=change_type,
            change_pct=change_pct,
            impact=impact
        ))
    
    # Sort method
    if before.sort_method or after.sort_method:
        before_sort = before.sort_method or "None"
        after_sort = after.sort_method or "None"
        
        # external/disk sort is bad
        before_disk = 'external' in before_sort.lower() or 'disk' in before_sort.lower()
        after_disk = 'external' in after_sort.lower() or 'disk' in after_sort.lower()
        
        if before_disk and not after_disk:
            change_type = ChangeType.IMPROVED.value
            impact = "🟢 Sort now in memory"
        elif not before_disk and after_disk:
            change_type = ChangeType.REGRESSED.value
            impact = "🔴 Sort now spills to disk"
        elif before_sort != after_sort:
            change_type = ChangeType.UNCHANGED.value
            impact = "Sort method changed"
        else:
            change_type = ChangeType.UNCHANGED.value
            impact = "Same sort method"
        
        comparisons.append(MetricComparison(
            metric="Sort Method",
            before=before_sort,
            after=after_sort,
            change_type=change_type,
            impact=impact
        ))
    
    # Warnings comparison
    before_warnings = set(before.warnings)
    after_warnings = set(after.warnings)
    
    removed_warnings = before_warnings - after_warnings
    added_warnings = after_warnings - before_warnings
    
    if removed_warnings:
        comparisons.append(MetricComparison(
            metric="Warnings Resolved",
            before=", ".join(removed_warnings),
            after="None",
            change_type=ChangeType.IMPROVED.value,
            impact="🟢 Issues fixed"
        ))
    
    if added_warnings:
        comparisons.append(MetricComparison(
            metric="New Warnings",
            before="None",
            after=", ".join(added_warnings),
            change_type=ChangeType.REGRESSED.value,
            impact="🔴 New issues introduced"
        ))
    
    return comparisons

def analyze_scan_changes(before: PlanMetrics, after: PlanMetrics) -> Dict[str, str]:
    """Analyze changes in scan types."""
    changes = {}
    
    before_scans = set(before.scan_types)
    after_scans = set(after.scan_types)
    
    # Rank scan types (lower is better)
    scan_rank = {
        'Seq Scan': 5,
        'Full Table Scan': 5,
        'Full Index Scan': 4,
        'Bitmap Heap Scan': 3,
        'Bitmap Index Scan': 2,
        'Index Range Scan': 2,
        'Index Scan': 1,
        'Index Lookup (ref)': 1,
        'Index Lookup (eq_ref)': 1,
        'Index Only Scan': 0,
        'Const Lookup': 0,
        'System Const': 0,
    }
    
    # Check for improvements
    for scan in before_scans - after_scans:
        changes[f"Removed: {scan}"] = "🟢 Eliminated"
    
    for scan in after_scans - before_scans:
        changes[f"Added: {scan}"] = "⚪ New"
    
    # Check if we moved to better scan types
    before_worst = max((scan_rank.get(s, 3) for s in before_scans), default=0)
    after_worst = max((scan_rank.get(s, 3) for s in after_scans), default=0)
    
    if before_worst > after_worst and before_worst >= 4:
        changes["Scan Upgrade"] = "🟢 Moved from full scan to index scan"
    elif after_worst > before_worst and after_worst >= 4:
        changes["Scan Downgrade"] = "🔴 Fell back to full scan"
    
    # Index usage
    if not before.index_names and after.index_names:
        changes["Index Usage"] = f"🟢 Now using indexes: {', '.join(after.index_names)}"
    elif before.index_names and not after.index_names:
        changes["Index Usage"] = "🔴 No longer using indexes"
    elif before.index_names != after.index_names:
        changes["Index Change"] = f"Changed from {before.index_names} to {after.index_names}"
    
    return changes

def generate_recommendations(before: PlanMetrics, after: PlanMetrics, comparisons: List[MetricComparison]) -> List[str]:
    """Generate recommendations based on the comparison."""
    recommendations = []
    
    # Check overall trend
    improved = sum(1 for c in comparisons if c.change_type == ChangeType.IMPROVED.value)
    regressed = sum(1 for c in comparisons if c.change_type == ChangeType.REGRESSED.value)
    
    if regressed > improved:
        recommendations.append("⚠️  Overall regression detected - consider reverting changes")
    
    # Specific recommendations
    if 'Seq Scan' in after.scan_types or 'Full Table Scan' in after.scan_types:
        recommendations.append("Consider adding index to eliminate full table scan")
    
    if after.sort_method and ('external' in after.sort_method.lower() or 'disk' in after.sort_method.lower()):
        recommendations.append("Sort spills to disk - increase work_mem or add index for ORDER BY")
    
    if after.shared_buffers_read and after.shared_buffers_hit:
        hit_ratio = after.shared_buffers_hit / (after.shared_buffers_hit + after.shared_buffers_read)
        if hit_ratio < 0.9:
            recommendations.append(f"Cache hit ratio is {hit_ratio:.1%} - consider increasing shared_buffers")
    
    if after.warnings:
        for warning in after.warnings:
            if 'estimate' in warning.lower():
                recommendations.append("Row estimates are off - run ANALYZE to update statistics")
    
    # If we have actual timing
    if after.execution_time_ms and after.execution_time_ms > 1000:
        recommendations.append("Query still takes >1s - may need further optimization")
    
    if not recommendations:
        if improved > 0:
            recommendations.append("✅ Optimization successful - consider deploying")
        else:
            recommendations.append("No significant change detected - verify the optimization was applied")
    
    return recommendations

def diff_explain(before_output: str, after_output: str, db_type: str) -> DiffResult:
    """Compare two EXPLAIN outputs and generate diff report."""
    
    # Parse both outputs
    if db_type == 'postgresql':
        before = parse_postgresql_explain(before_output)
        after = parse_postgresql_explain(after_output)
    else:
        before = parse_mysql_explain(before_output)
        after = parse_mysql_explain(after_output)
    
    # Compare metrics
    comparisons = compare_metrics(before, after)
    
    # Analyze scan changes
    scan_changes = analyze_scan_changes(before, after)
    
    # Generate recommendations
    recommendations = generate_recommendations(before, after, comparisons)
    
    # Determine overall verdict
    improved = sum(1 for c in comparisons if c.change_type == ChangeType.IMPROVED.value)
    regressed = sum(1 for c in comparisons if c.change_type == ChangeType.REGRESSED.value)
    
    if improved > 0 and regressed == 0:
        verdict = "improved"
        summary = f"✅ Query improved in {improved} metric(s)"
    elif regressed > 0 and improved == 0:
        verdict = "regressed"
        summary = f"❌ Query regressed in {regressed} metric(s)"
    elif improved > 0 and regressed > 0:
        verdict = "mixed"
        summary = f"⚠️ Mixed results: {improved} improved, {regressed} regressed"
    else:
        verdict = "unchanged"
        summary = "No significant changes detected"
    
    return DiffResult(
        database=db_type,
        summary=summary,
        overall_verdict=verdict,
        comparisons=[asdict(c) for c in comparisons],
        scan_type_changes=scan_changes,
        recommendations=recommendations
    )

def main():
    parser = argparse.ArgumentParser(description='Compare EXPLAIN outputs')
    parser.add_argument('before_file', nargs='?', help='Before EXPLAIN output file')
    parser.add_argument('after_file', nargs='?', help='After EXPLAIN output file')
    parser.add_argument('--before', help='Before EXPLAIN output as string')
    parser.add_argument('--after', help='After EXPLAIN output as string')
    parser.add_argument('--pg', '--postgresql', action='store_true', help='PostgreSQL format')
    parser.add_argument('--mysql', action='store_true', help='MySQL format')
    parser.add_argument('--json', '-j', action='store_true', help='Output as JSON')
    args = parser.parse_args()
    
    if not args.pg and not args.mysql:
        print("Error: Specify --pg or --mysql", file=sys.stderr)
        sys.exit(1)
    
    # Get before output
    if args.before:
        before_output = args.before
    elif args.before_file:
        try:
            with open(args.before_file) as f:
                before_output = f.read()
        except FileNotFoundError:
            print(f"Error: File '{args.before_file}' not found", file=sys.stderr)
            sys.exit(1)
    else:
        print("Error: Provide before output via file or --before", file=sys.stderr)
        sys.exit(1)
    
    # Get after output
    if args.after:
        after_output = args.after
    elif args.after_file:
        try:
            with open(args.after_file) as f:
                after_output = f.read()
        except FileNotFoundError:
            print(f"Error: File '{args.after_file}' not found", file=sys.stderr)
            sys.exit(1)
    else:
        print("Error: Provide after output via file or --after", file=sys.stderr)
        sys.exit(1)
    
    db_type = 'postgresql' if args.pg else 'mysql'
    result = diff_explain(before_output, after_output, db_type)
    
    if args.json:
        print(json.dumps(asdict(result), indent=2))
    else:
        print(f"{'='*60}")
        print(f"EXPLAIN DIFF - {result.database.upper()}")
        print(f"{'='*60}")
        print(f"\n{result.summary}\n")
        
        if result.comparisons:
            print("METRIC COMPARISON")
            print("-" * 60)
            for comp in result.comparisons:
                print(f"\n{comp['metric']}:")
                print(f"  Before: {comp['before']}")
                print(f"  After:  {comp['after']}")
                print(f"  {comp['impact']}")
        
        if result.scan_type_changes:
            print(f"\n{'SCAN TYPE CHANGES'}")
            print("-" * 60)
            for change, status in result.scan_type_changes.items():
                print(f"  {status} {change}")
        
        if result.recommendations:
            print(f"\n{'RECOMMENDATIONS'}")
            print("-" * 60)
            for rec in result.recommendations:
                print(f"  • {rec}")
        
        print()

if __name__ == "__main__":
    main()
