#!/usr/bin/env python3
"""
Parse and analyze database slow query logs.
Supports PostgreSQL (log_min_duration_statement) and MySQL slow query log.

Usage: python parse_slow_log.py --pg postgresql.log
       python parse_slow_log.py --mysql slow-query.log --top 20
       python parse_slow_log.py --pg postgresql.log --json
"""

import re
import sys
import json
import argparse
from collections import defaultdict
from dataclasses import dataclass, asdict
from typing import List, Dict, Optional
from datetime import datetime
import hashlib

@dataclass
class SlowQuery:
    query_hash: str
    normalized_query: str
    sample_query: str
    count: int
    total_time_ms: float
    avg_time_ms: float
    max_time_ms: float
    min_time_ms: float
    timestamps: List[str]

@dataclass
class LogAnalysis:
    database_type: str
    total_queries: int
    unique_queries: int
    total_time_ms: float
    time_range: Dict[str, str]
    slow_queries: List[SlowQuery]
    summary: str

def normalize_query(query: str) -> str:
    """Normalize query by replacing literals with placeholders."""
    # Remove extra whitespace
    normalized = re.sub(r'\s+', ' ', query.strip())
    
    # Replace string literals
    normalized = re.sub(r"'[^']*'", "'?'", normalized)
    normalized = re.sub(r'"[^"]*"', '"?"', normalized)
    
    # Replace numeric literals (but not in identifiers)
    normalized = re.sub(r'\b\d+\b', '?', normalized)
    
    # Replace IN lists
    normalized = re.sub(r'IN\s*\(\s*\?\s*(?:,\s*\?\s*)+\)', 'IN (?)', normalized, flags=re.IGNORECASE)
    
    # Replace LIMIT/OFFSET values
    normalized = re.sub(r'(LIMIT|OFFSET)\s+\?', r'\1 ?', normalized, flags=re.IGNORECASE)
    
    return normalized

def query_hash(normalized: str) -> str:
    """Generate a short hash for query identification."""
    return hashlib.md5(normalized.encode()).hexdigest()[:12]

def parse_postgresql_log(content: str) -> List[Dict]:
    """Parse PostgreSQL log format."""
    queries = []
    
    # Pattern for log_min_duration_statement output
    # Example: 2024-01-15 10:30:45.123 UTC [12345] LOG:  duration: 1234.567 ms  statement: SELECT ...
    pattern = r'(\d{4}-\d{2}-\d{2}\s+\d{2}:\d{2}:\d{2}[.\d]*)\s+\w+\s+\[\d+\].*?duration:\s*([\d.]+)\s*ms\s+(?:statement|execute\s+\w+):\s*(.+?)(?=\n\d{4}-\d{2}-\d{2}|\Z)'
    
    for match in re.finditer(pattern, content, re.DOTALL | re.IGNORECASE):
        timestamp = match.group(1)
        duration = float(match.group(2))
        query = match.group(3).strip()
        
        # Clean up multi-line queries
        query = re.sub(r'\s+', ' ', query)
        
        queries.append({
            'timestamp': timestamp,
            'duration_ms': duration,
            'query': query
        })
    
    # Alternative pattern for simpler log format
    if not queries:
        pattern = r'duration:\s*([\d.]+)\s*ms\s+(?:statement|execute):\s*(.+?)(?=duration:|\Z)'
        for match in re.finditer(pattern, content, re.DOTALL | re.IGNORECASE):
            queries.append({
                'timestamp': '',
                'duration_ms': float(match.group(1)),
                'query': re.sub(r'\s+', ' ', match.group(2).strip())
            })
    
    return queries

def parse_mysql_slow_log(content: str) -> List[Dict]:
    """Parse MySQL slow query log format."""
    queries = []
    
    # Split by query entries
    # MySQL slow log format:
    # # Time: 2024-01-15T10:30:45.123456Z
    # # User@Host: user[user] @ localhost []
    # # Query_time: 1.234567  Lock_time: 0.000123 Rows_sent: 100  Rows_examined: 10000
    # SET timestamp=1705314645;
    # SELECT ...;
    
    entries = re.split(r'(?=# Time:)', content)
    
    for entry in entries:
        if not entry.strip():
            continue
        
        # Extract time
        time_match = re.search(r'# Time:\s*(\S+)', entry)
        timestamp = time_match.group(1) if time_match else ''
        
        # Extract query time
        qt_match = re.search(r'Query_time:\s*([\d.]+)', entry)
        if not qt_match:
            continue
        duration_ms = float(qt_match.group(1)) * 1000  # Convert to ms
        
        # Extract query (after SET timestamp line or after the metadata)
        query_match = re.search(r'(?:SET timestamp=\d+;\s*)?([A-Z][\s\S]+?);?\s*$', entry, re.IGNORECASE)
        if query_match:
            query = re.sub(r'\s+', ' ', query_match.group(1).strip())
            queries.append({
                'timestamp': timestamp,
                'duration_ms': duration_ms,
                'query': query
            })
    
    # Fallback: simpler pattern
    if not queries:
        pattern = r'Query_time:\s*([\d.]+).*?(?:SET timestamp=\d+;)?\s*(\b(?:SELECT|INSERT|UPDATE|DELETE|CALL)\b[^;]+)'
        for match in re.finditer(pattern, content, re.DOTALL | re.IGNORECASE):
            queries.append({
                'timestamp': '',
                'duration_ms': float(match.group(1)) * 1000,
                'query': re.sub(r'\s+', ' ', match.group(2).strip())
            })
    
    return queries

def aggregate_queries(raw_queries: List[Dict]) -> List[SlowQuery]:
    """Aggregate queries by normalized form."""
    aggregated: Dict[str, Dict] = defaultdict(lambda: {
        'normalized': '',
        'sample': '',
        'count': 0,
        'total_time': 0,
        'max_time': 0,
        'min_time': float('inf'),
        'timestamps': []
    })
    
    for q in raw_queries:
        normalized = normalize_query(q['query'])
        h = query_hash(normalized)
        
        agg = aggregated[h]
        agg['normalized'] = normalized
        if not agg['sample']:
            agg['sample'] = q['query'][:500]  # Keep first sample
        agg['count'] += 1
        agg['total_time'] += q['duration_ms']
        agg['max_time'] = max(agg['max_time'], q['duration_ms'])
        agg['min_time'] = min(agg['min_time'], q['duration_ms'])
        if q['timestamp']:
            agg['timestamps'].append(q['timestamp'])
    
    result = []
    for h, agg in aggregated.items():
        result.append(SlowQuery(
            query_hash=h,
            normalized_query=agg['normalized'],
            sample_query=agg['sample'],
            count=agg['count'],
            total_time_ms=round(agg['total_time'], 2),
            avg_time_ms=round(agg['total_time'] / agg['count'], 2),
            max_time_ms=round(agg['max_time'], 2),
            min_time_ms=round(agg['min_time'], 2) if agg['min_time'] != float('inf') else 0,
            timestamps=agg['timestamps'][:5]  # Keep first 5 timestamps
        ))
    
    return result

def analyze_log(raw_queries: List[Dict], db_type: str, top_n: int) -> LogAnalysis:
    """Analyze parsed queries and generate report."""
    if not raw_queries:
        return LogAnalysis(
            database_type=db_type,
            total_queries=0,
            unique_queries=0,
            total_time_ms=0,
            time_range={},
            slow_queries=[],
            summary="No slow queries found in log."
        )
    
    aggregated = aggregate_queries(raw_queries)
    
    # Sort by total time (biggest impact first)
    aggregated.sort(key=lambda x: x.total_time_ms, reverse=True)
    
    total_time = sum(q['duration_ms'] for q in raw_queries)
    
    # Time range
    timestamps = [q['timestamp'] for q in raw_queries if q['timestamp']]
    time_range = {}
    if timestamps:
        time_range = {'start': min(timestamps), 'end': max(timestamps)}
    
    # Summary
    top_query_time = aggregated[0].total_time_ms if aggregated else 0
    top_query_pct = (top_query_time / total_time * 100) if total_time > 0 else 0
    
    summary = f"Analyzed {len(raw_queries)} slow queries ({len(aggregated)} unique patterns). "
    summary += f"Total time: {total_time/1000:.1f}s. "
    if top_query_pct > 50:
        summary += f"Top query accounts for {top_query_pct:.0f}% of total time - prioritize this!"
    
    return LogAnalysis(
        database_type=db_type,
        total_queries=len(raw_queries),
        unique_queries=len(aggregated),
        total_time_ms=round(total_time, 2),
        time_range=time_range,
        slow_queries=[asdict(q) for q in aggregated[:top_n]],
        summary=summary
    )

def main():
    parser = argparse.ArgumentParser(description='Parse slow query logs')
    parser.add_argument('file', nargs='?', help='Log file to analyze')
    parser.add_argument('--pg', '--postgresql', action='store_true', help='PostgreSQL log format')
    parser.add_argument('--mysql', action='store_true', help='MySQL slow query log format')
    parser.add_argument('--top', '-n', type=int, default=10, help='Number of top queries to show (default: 10)')
    parser.add_argument('--json', '-j', action='store_true', help='Output as JSON')
    args = parser.parse_args()
    
    if not args.pg and not args.mysql:
        print("Error: Specify --pg or --mysql log format", file=sys.stderr)
        sys.exit(1)
    
    if args.file:
        try:
            with open(args.file) as f:
                content = f.read()
        except FileNotFoundError:
            print(f"Error: File '{args.file}' not found", file=sys.stderr)
            sys.exit(1)
    else:
        content = sys.stdin.read()
    
    if not content.strip():
        print("Error: No log content provided", file=sys.stderr)
        sys.exit(1)
    
    # Parse based on database type
    if args.pg:
        raw_queries = parse_postgresql_log(content)
        db_type = 'postgresql'
    else:
        raw_queries = parse_mysql_slow_log(content)
        db_type = 'mysql'
    
    result = analyze_log(raw_queries, db_type, args.top)
    
    if args.json:
        print(json.dumps(asdict(result), indent=2))
    else:
        print(f"Database: {result.database_type.upper()}")
        print(f"Summary: {result.summary}")
        print(f"\nTotal queries: {result.total_queries}")
        print(f"Unique patterns: {result.unique_queries}")
        print(f"Total time: {result.total_time_ms/1000:.2f}s")
        
        if result.time_range:
            print(f"Time range: {result.time_range.get('start', 'N/A')} to {result.time_range.get('end', 'N/A')}")
        
        print(f"\n{'='*60}")
        print(f"Top {len(result.slow_queries)} Slow Query Patterns (by total time)")
        print(f"{'='*60}\n")
        
        for i, q in enumerate(result.slow_queries, 1):
            pct = (q['total_time_ms'] / result.total_time_ms * 100) if result.total_time_ms > 0 else 0
            print(f"{i}. [{q['query_hash']}] {pct:.1f}% of total time")
            print(f"   Count: {q['count']} | Avg: {q['avg_time_ms']:.1f}ms | Max: {q['max_time_ms']:.1f}ms")
            print(f"   Query: {q['normalized_query'][:200]}{'...' if len(q['normalized_query']) > 200 else ''}")
            print()

if __name__ == "__main__":
    main()
