# 1.2 Extract to Memoized Components

Extract expensive work into memoized components to enable early returns before computation.

**❌ Incorrect: computes avatar even when loading**
```tsx
function Profile({ user, loading }: Props) {
  const avatar = useMemo(() => {
    const id = computeAvatarId(user);
    return <Avatar id={id} />;
  }, [user]);

  if (loading) {
    return <Skeleton />;
  }

  return <div>{avatar}</div>;
}
```

**✅ Correct: skips computation when loading**
```tsx
const UserAvatar = memo(function UserAvatar({ user }: { user: User }) {
  const id = useMemo(() => computeAvatarId(user), [user]);
  return <Avatar id={id} />;
})

function Profile({ user, loading }: Props) {
  if (loading) {
    return <Skeleton />;
  }

  return (
    <div>
      <UserAvatar user={user} />
    </div>
  )
}
```

---

## React Compiler Note

✅ **Handled automatically** - If your project has [React Compiler](https://react.dev/learn/react-compiler) enabled, manual memoization with `memo()` and `useMemo()` is unnecessary. The compiler automatically optimizes re-renders.

See [react-compiler-guide.md](react-compiler-guide.md) for more details.
