# 4.9 Cache Storage API Calls

`localStorage`, `sessionStorage`, and `document.cookie` are synchronous and expensive. Cache reads in memory.

**❌ Incorrect: reads storage on every call**
```ts
function getTheme() {
  return localStorage.getItem('theme') ?? 'light';
}
// Called 10 times = 10 storage reads
```

**✅ Correct: `Map` cache**
```ts
const storageCache = new Map<string, string | null>()

function getLocalStorage(key: string) {
  if (!storageCache.has(key)) {
    storageCache.set(key, localStorage.getItem(key));
  }

  return storageCache.get(key);
}

function setLocalStorage(key: string, value: string) {
  localStorage.setItem(key, value);
  storageCache.set(key, value);  // keep cache in sync
}
```

Cookie caching:

```ts
let cookieCache: Record<string, string> | null = null

function getCookie(name: string) {
  if (!cookieCache) {
    cookieCache = Object.fromEntries(
      document.cookie.split('; ').map(c => c.split('='));
    )
  }

  return cookieCache[name];
}
```

**Important**: invalidate on external changes

```ts
window.addEventListener('storage', (e) => {
  if (e.key) {
    storageCache.delete(e.key);
  }
});

document.addEventListener('visibilitychange', () => {
  if (document.visibilityState === 'visible') {
    storageCache.clear()
  }
});
```

If storage can change externally (another tab, server-set cookies), invalidate cache:
