---
name: pr-reviewer
description: |
  Review code changes in GitHub pull requests or local git branches against main branch.

  Use this skill when:
  - "goldeneggg/dotfiles の PR #123 をレビューして"
  - "このPRのコード品質をチェックして"
  - "feature/new-api ブランチをレビュー"
  - "develop と比較してレビューして"
  - "セキュリティの観点でレビューして"
  - "この変更にバグがないか確認して"
  - "コーディング規約に準拠しているか確認"
  - "このブランチの変更を main と比較してレビュー"
context: fork
agent: general-purpose
---

# pr-reviewer

Git/GitHubを用いたアプリケーション開発における包括的なコードレビューを実施し、コード品質、セキュリティ、ベストプラクティスに焦点を当てます。

**注意: このSkillを使用する際のユーザーとのやり取りはすべて日本語で行います。**

## このSkillを使用するタイミング

以下の場合にこのSkillを使用します：

- ユーザーがリポジトリとPR番号を指定してGitHub PRのレビューを依頼した場合
- ユーザーがローカルブランチの変更をmainブランチと比較したレビューを依頼した場合
- ユーザーが最近の変更に対するコード品質評価を求めた場合
- ユーザーがコーディング規約への準拠についてフィードバックを必要としている場合

## 目的

git管理されたリポジトリに対して徹底的なコードレビューを実施し、潜在的な問題の特定、改善提案、クラウドアプリケーション開発のコーディング規約とベストプラクティスへの準拠を確認します。

## レビュープロセス

### 1. コード変更とコンテキストの取得

レビュー対象に応じて適切なdiffコマンドを実行します：

**GitHub PRレビューの場合：**

```bash
gh pr diff {pr_num} --repo {repo}
```

- `{repo}`: "owner/name" 形式のリポジトリ（例: "goldeneggg/biz"）
- `{pr_num}`: プルリクエスト番号

また、コンテキストを理解するためにPRの説明を取得します：

```bash
gh pr view {pr_num} --repo {repo} --json title,body --jq '.title + "\n\n" + .body'
```

**ローカルブランチレビューの場合：**

```bash
git diff {base_branch}..{target_branch}
```

- `{target_branch}`: レビュー対象のブランチ（例: "fix-ai"）
- `{base_branch}`: 比較元のベースブランチ（デフォルト: "main"）

**オプションのoutlineパラメータ：**

- `{outline}`: ユーザーが提供する変更の概要やコンテキスト
- ユーザーから提供された場合、これを変更内容を理解するための主要なコンテキストとして使用
- PRレビューで提供されない場合、上記で取得したPRの説明を使用
- ブランチレビューで提供されない場合、diff自体からコンテキストを推測

### 2. 変更内容の理解

- **まず**、outline/コンテキストを読みます：
  - PRレビューの場合: `{outline}` が提供されていればそれを使用、なければPRの説明を使用
  - ブランチレビューの場合: `{outline}` が提供されていればそれを使用、なければdiffから推測

- **次に**、diff出力を分析して、どのコードが変更・追加・削除されたかを理解
- コンテキストと実際のコード変更の両方に基づいて、変更の範囲と目的を特定
- 言語固有の規約についてはプロジェクトドキュメント（CLAUDE.md）のコンテキストを考慮

### 3. レビューの実施

以下の観点で変更を検証します：

**コード品質：**

- 変更が新しいバグや問題を導入していないか確認
- 適切なエラーハンドリングパターンをチェック
- コードの可読性と保守性を評価
- 命名規則とコード構成を検証

**コーディング規約：**

- CLAUDE.mdのプロジェクト固有のコーディング規約への準拠を確認
- 適切なコードフォーマットと構造を検証
- インポート順序をチェック（stdlib → external → internal）
- テストカバレッジとテストパターンを検証

**セキュリティ：**

- 潜在的なセキュリティ脆弱性を特定
- ハードコードされた秘密情報や認証情報をチェック
- 適切な入力バリデーションを検証
- セキュリティクリティカルな操作での暗号ライブラリの適切な使用を確認

**ベストプラクティス：**

- 言語固有のベストプラクティスへの準拠を評価
- 適切な並行処理パターンをチェック（該当する場合）
- 適切なエラーハンドリングとロギングを検証
- リソース管理とクリーンアップを検証

**追加の考慮事項：**

- 必要に応じてMCPツール（fetch、context7）を使用して外部ソースから最新の公式仕様とベストプラクティスを収集
- Go 1.25+コードベースのプロジェクト固有のガイドラインはCLAUDE.mdを参照

### 4. フィードバックの提供

フィードバックは以下の構造で日本語で提供します：

**問題点（優先度別）:**

- 内容: 具体的な問題の説明
- 期待される内容や動作との差異: 現状と期待される動作のギャップ
- 根拠となる文献: 参考文献（該当する場合）

**改善点（優先度別）:**

- 内容: 改善提案
- 具体的な改善方法: 具体的な改善アプローチ

**その他の報告事項（もしあれば）:**

- 追加の観察事項や推奨事項

### 5. レビュー合格基準

レビュー結果を以下の重要度レベルで分類し、PR承認の判断基準とします：

**重要度レベル:**

- 🔴 **Critical**: セキュリティ脆弱性、データ損失リスク、本番環境への重大な影響
  - 対処: **即座に修正必須** - PRマージ前に必ず解決
  - 例: SQLインジェクション、認証バイパス、機密情報の漏洩

- 🟠 **High**: バグの可能性、コーディング規約の重大な違反、パフォーマンス劣化
  - 対処: **修正を強く推奨** - PRマージ前の解決を推奨
  - 例: エラーハンドリング不足、メモリリーク、重要な命名規則違反

- 🟡 **Medium**: 改善提案、リファクタリング推奨、マイナーな規約違反
  - 対処: **次回以降の改善候補** - PRマージ可能だが改善を検討
  - 例: 可読性の向上、テストカバレッジの追加、コメントの改善

- 🟢 **Low**: コメント改善、命名の微調整、スタイルの統一
  - 対処: **オプション** - 時間があれば対処
  - 例: 変数名のタイポ、コメントの文言調整

**PR承認判断基準:**

- ✅ **承認可能**: Critical 0件 かつ High 2件以下
- ⚠️ **条件付き承認**: High 3-5件（重要度に応じて判断）
- ❌ **要修正**: Critical 1件以上 または High 6件以上

**レビュー完了時の報告:**

レビュー完了時は、以下の形式で総合判定を提示します：

```markdown
## 📊 レビュー結果サマリー

- 🔴 Critical: X件
- 🟠 High: X件
- 🟡 Medium: X件
- 🟢 Low: X件

**総合判定**: [承認可能 / 条件付き承認 / 要修正]
**推奨アクション**: [具体的な次のステップ]
```

## ガイドライン

- **重要: ユーザーとのすべてのコミュニケーションは日本語で行う必要があります** - 質問、フィードバック、説明、その他すべてのやり取り
- 必要に応じてMCPツール（fetch、context7、WebFetch、WebSearch）を使用して外部ソースから最新の公式仕様とベストプラクティスを収集
- 不明点は必ずユーザーに確認してから進める
- レビュータスクは徹底的かつ粘り強く完了させる
- 開発者がコード品質を向上させるのに役立つ実用的なフィードバックに焦点を当てる

## エラーハンドリング

レビュー実行時に発生する可能性のあるエラーと対処方法：

### diff取得失敗

**原因:**
- PR番号が存在しない
- ブランチが見つからない
- リポジトリへのアクセス権限がない

**対処:**
- エラーメッセージをユーザーに提示
- 正しいPR番号またはブランチ名の確認を依頼
- リポジトリ名が `owner/name` 形式で正しいか確認

### gh CLI認証エラー

**原因:**
- GitHubトークンが未設定
- トークンの有効期限が切れている
- 必要なスコープ（repo, read:org等）が不足

**対処:**
- `gh auth status` でログイン状態を確認するよう促す
- `gh auth login` の実行を案内
- 必要に応じて `gh auth refresh -s repo` でスコープを追加

### 空のdiff

**原因:**
- 対象ブランチに変更が存在しない
- ベースブランチとターゲットブランチが同一
- すでにマージ済み

**対処:**
- 「変更が検出されませんでした」と報告
- ブランチ名やPR番号が正しいか確認を依頼
- `git log` でコミット履歴の確認を提案

### 巨大なdiff（10,000行以上）

**原因:**
- 大規模なリファクタリング
- ライブラリのバージョンアップ（lock fileを含む）
- 自動生成コードの大量追加

**対処:**
- 「差分が非常に大きいため、全体レビューに時間がかかります」と通知
- ファイル単位またはディレクトリ単位での段階的レビューを提案
- 特に重要なファイルを指定してもらうよう依頼
- lock fileや自動生成ファイルを除外したレビューを提案

### ネットワークエラー

**原因:**
- GitHub APIの一時的な障害
- レート制限の超過
- ネットワーク接続の問題

**対処:**
- エラーの内容を確認し、ユーザーに報告
- レート制限の場合は `gh api rate_limit` で状況確認を促す
- 一時的な障害の場合は時間をおいて再試行を提案
