# Automation Patterns

Automation patterns for integrating Claude Code into your development workflow, including pre-commit hooks, scheduled tasks, and event-driven triggers.

## Pre-commit Hooks

### Using pre-commit Framework

Install pre-commit:
```bash
pip install pre-commit
# or
brew install pre-commit
```

Create `.pre-commit-config.yaml`:

```yaml
repos:
  - repo: local
    hooks:
      - id: claude-quick-review
        name: Claude Quick Review
        entry: bash -c 'git diff --cached | claude -p "Quick review of staged changes. Only report critical issues (security, obvious bugs). Be brief." --max-turns 1 --output-format text'
        language: system
        stages: [pre-commit]
        pass_filenames: false
        verbose: true
```

Install hooks:
```bash
pre-commit install
```

### Native Git Hooks

Create `.git/hooks/pre-commit`:

```bash
#!/bin/bash
set -e

# Check if there are staged changes
staged=$(git diff --cached --name-only)
if [ -z "$staged" ]; then
  exit 0
fi

# Get staged diff
diff=$(git diff --cached)

# Quick Claude review
echo "Running Claude quick review..."
result=$(echo "$diff" | claude -p "Quick review of these changes. Report ONLY:
- Security vulnerabilities
- Obvious bugs that will crash
- Secrets/credentials in code

If nothing critical, respond with just 'OK'.
Be extremely brief." --max-turns 1 --output-format text 2>&1)

# Check for critical issues
if echo "$result" | grep -qiE "(security|vulnerability|credential|secret|crash|critical)"; then
  echo "Claude found potential issues:"
  echo "$result"
  echo ""
  read -p "Continue anyway? (y/N) " -n 1 -r
  echo
  if [[ ! $REPLY =~ ^[Yy]$ ]]; then
    exit 1
  fi
fi

echo "Claude review: OK"
exit 0
```

Make executable:
```bash
chmod +x .git/hooks/pre-commit
```

### Pre-push Hook

Create `.git/hooks/pre-push`:

```bash
#!/bin/bash
set -e

# Get commits being pushed
remote="$1"
url="$2"

while read local_ref local_sha remote_ref remote_sha; do
  if [ "$local_sha" = "0000000000000000000000000000000000000000" ]; then
    continue  # Deleting branch
  fi

  if [ "$remote_sha" = "0000000000000000000000000000000000000000" ]; then
    # New branch - review all commits
    range="$local_sha"
  else
    # Existing branch - review new commits only
    range="$remote_sha..$local_sha"
  fi

  # Get changes
  changes=$(git diff $range)

  if [ -n "$changes" ]; then
    echo "Running Claude pre-push review..."
    claude -p "Review these changes about to be pushed:

$changes

Check for:
1. Secrets or credentials
2. Debug code (console.log, debugger)
3. Test-only code in production
4. Incomplete implementations (TODO, FIXME in new code)

Report only blocking issues." --max-turns 2 --output-format text
  fi
done

exit 0
```

### Commit Message Hook

Create `.git/hooks/prepare-commit-msg`:

```bash
#!/bin/bash
COMMIT_MSG_FILE=$1
COMMIT_SOURCE=$2

# Skip if commit message already provided (amend, merge, etc.)
if [ -n "$COMMIT_SOURCE" ]; then
  exit 0
fi

# Get staged diff
diff=$(git diff --cached)

if [ -z "$diff" ]; then
  exit 0
fi

echo "Generating commit message..."

# Generate commit message
message=$(echo "$diff" | claude -p "Generate a conventional commit message for this diff.
Format: type(scope): description

Types: feat, fix, docs, style, refactor, test, chore
Keep under 72 chars. Be specific." --max-turns 1 --output-format text)

# Prepend generated message
echo "$message" > "$COMMIT_MSG_FILE.tmp"
echo "" >> "$COMMIT_MSG_FILE.tmp"
echo "# Generated by Claude Code" >> "$COMMIT_MSG_FILE.tmp"
echo "# Edit as needed or leave as-is" >> "$COMMIT_MSG_FILE.tmp"
cat "$COMMIT_MSG_FILE" >> "$COMMIT_MSG_FILE.tmp"
mv "$COMMIT_MSG_FILE.tmp" "$COMMIT_MSG_FILE"
```

## Scheduled Tasks

### Cron-based Local Automation

Using crontab for scheduled reviews:

```bash
# Edit crontab
crontab -e

# Add daily code review at 9 AM
0 9 * * * cd /path/to/project && /path/to/review-script.sh >> /var/log/claude-review.log 2>&1
```

Review script (`review-script.sh`):

```bash
#!/bin/bash
set -e

PROJECT_DIR="/path/to/project"
cd "$PROJECT_DIR"

# Check for changes since last review
last_review=$(cat .claude-last-review 2>/dev/null || echo "HEAD~10")
changes=$(git diff $last_review..HEAD --name-only)

if [ -z "$changes" ]; then
  echo "$(date): No changes to review"
  exit 0
fi

echo "$(date): Reviewing changes since $last_review"

# Run review
claude -p "Daily code review of recent changes:

Changed files:
$changes

Full diff:
$(git diff $last_review..HEAD)

Provide:
1. Summary of changes
2. Potential issues
3. Suggestions for improvement" \
  --allowedTools "Read,Glob,Grep" \
  --max-turns 5 \
  --output-format text > daily-review.md

# Update last review marker
git rev-parse HEAD > .claude-last-review

# Optionally notify via webhook
if [ -n "$SLACK_WEBHOOK" ]; then
  curl -X POST "$SLACK_WEBHOOK" \
    -H "Content-Type: application/json" \
    -d "{\"text\": \"Daily code review completed. See daily-review.md\"}"
fi
```

### Scheduled Dependency Updates

```bash
#!/bin/bash
# weekly-deps.sh - Run weekly for dependency updates

cd /path/to/project

# Check for outdated packages
echo "Checking for outdated packages..."
outdated=$(npm outdated --json 2>/dev/null || echo "{}")

if [ "$outdated" = "{}" ]; then
  echo "All packages up to date"
  exit 0
fi

# Analyze updates
claude -p "Analyze these npm package updates:

$outdated

For each update:
1. Check if it's a security update (prioritize)
2. Check for breaking changes
3. Recommend update strategy (immediate, scheduled, skip)

Output a prioritized update plan." \
  --max-turns 3 \
  --output-format text > update-plan.md

echo "Update plan generated: update-plan.md"
```

### Scheduled Health Checks

```bash
#!/bin/bash
# health-check.sh - Run weekly

cd /path/to/project

claude -p "Perform a codebase health check:

1. Find all TODOs, FIXMEs, HACKs
2. Identify dead code (unused exports, unreachable code)
3. Check for common anti-patterns
4. Review test coverage gaps
5. Identify potential security issues

Provide actionable recommendations." \
  --allowedTools "Read,Glob,Grep" \
  --max-turns 10 \
  --output-format text > health-report.md
```

## Event-Driven Triggers

### File Watcher Automation

Using `fswatch` (macOS) or `inotifywait` (Linux):

```bash
#!/bin/bash
# watch-and-review.sh

PROJECT_DIR="/path/to/project"

fswatch -o "$PROJECT_DIR/src" | while read num; do
  echo "Changes detected, running review..."

  # Debounce - wait for changes to settle
  sleep 2

  # Get recent changes
  changed=$(find "$PROJECT_DIR/src" -type f -mmin -1 -name "*.ts")

  if [ -n "$changed" ]; then
    for file in $changed; do
      claude -p "Quick review of recent changes to $file. Report only critical issues." \
        --allowedTools "Read" \
        --max-turns 1
    done
  fi
done
```

### GitHub Webhook Handler

Simple webhook handler for custom automation:

```typescript
// webhook-handler.ts
import { serve } from "bun";
import { execSync } from "child_process";

serve({
  port: 3000,
  async fetch(req) {
    if (req.method !== "POST") {
      return new Response("Method not allowed", { status: 405 });
    }

    const event = req.headers.get("X-GitHub-Event");
    const payload = await req.json();

    if (event === "pull_request" && payload.action === "opened") {
      const prNumber = payload.number;
      const repo = payload.repository.full_name;

      // Run async review
      execSync(`
        cd /path/to/repo &&
        gh pr diff ${prNumber} | claude -p "Review this PR" --max-turns 3
      `, { stdio: "inherit" });
    }

    return new Response("OK");
  },
});
```

### IDE Integration Triggers

VS Code task for on-save review:

```json
// .vscode/tasks.json
{
  "version": "2.0.0",
  "tasks": [
    {
      "label": "Claude Review Current File",
      "type": "shell",
      "command": "claude -p \"Review ${file} for issues\" --allowedTools Read --max-turns 2",
      "presentation": {
        "reveal": "silent",
        "panel": "shared"
      },
      "runOptions": {
        "runOn": "folderOpen"
      }
    }
  ]
}
```

## Automation Scripts

### Batch File Review

```bash
#!/bin/bash
# batch-review.sh - Review multiple files

files=("$@")

if [ ${#files[@]} -eq 0 ]; then
  echo "Usage: batch-review.sh file1 file2 ..."
  exit 1
fi

for file in "${files[@]}"; do
  echo "Reviewing $file..."
  claude -p "Review $file for:
- Code quality issues
- Potential bugs
- Performance concerns

Be concise." \
    --allowedTools "Read" \
    --max-turns 2 \
    --output-format text

  echo "---"
done
```

### PR Preparation Script

```bash
#!/bin/bash
# prepare-pr.sh - Prepare a PR with Claude assistance

# Get current branch changes
base_branch=${1:-main}
current_branch=$(git branch --show-current)

if [ "$current_branch" = "$base_branch" ]; then
  echo "Already on $base_branch, nothing to prepare"
  exit 1
fi

# Get diff
diff=$(git diff $base_branch...$current_branch)

# Generate PR description
echo "Generating PR description..."
description=$(echo "$diff" | claude -p "Generate a PR description for these changes:

Include:
## Summary
Brief description of what changed and why

## Changes
- Bulleted list of specific changes

## Testing
- How to test these changes

## Checklist
- [ ] Tests added/updated
- [ ] Documentation updated
- [ ] No breaking changes (or documented)

Be concise and specific." --max-turns 2 --output-format text)

# Self-review
echo "Running self-review..."
review=$(echo "$diff" | claude -p "Self-review these changes before PR:

Check for:
1. Debug code left in
2. Console.log statements
3. Commented-out code
4. Incomplete implementations
5. Missing error handling

List any issues found." --max-turns 2 --output-format text)

echo ""
echo "=== PR DESCRIPTION ==="
echo "$description"
echo ""
echo "=== SELF-REVIEW ==="
echo "$review"
echo ""
echo "Ready to create PR? Run:"
echo "gh pr create --title \"...\" --body \"\$description\""
```

### Automated Refactoring

```bash
#!/bin/bash
# auto-refactor.sh - Automated refactoring with Claude

target=${1:-.}
pattern=${2:-"*.ts"}

echo "Finding files to refactor..."
files=$(find "$target" -name "$pattern" -type f)

for file in $files; do
  echo "Analyzing $file..."

  # Check if refactoring needed
  needs_refactor=$(claude -p "Analyze $file. Does it need refactoring?
  Respond with just YES or NO.

  Check for:
  - Functions over 50 lines
  - Deep nesting (> 3 levels)
  - Duplicate code
  - Complex conditionals" \
    --allowedTools "Read" \
    --max-turns 1 \
    --output-format text)

  if echo "$needs_refactor" | grep -qi "YES"; then
    echo "Refactoring $file..."
    claude -p "Refactor $file to improve:
    - Extract long functions
    - Reduce nesting
    - Remove duplication
    - Simplify conditionals

    Make incremental, safe changes." \
      --allowedTools "Read,Edit" \
      --max-turns 5
  fi
done
```

## Notification Integration

### Slack Notifications

```bash
#!/bin/bash
# notify-slack.sh

SLACK_WEBHOOK="$SLACK_WEBHOOK_URL"

send_slack() {
  local message="$1"
  curl -X POST "$SLACK_WEBHOOK" \
    -H "Content-Type: application/json" \
    -d "{\"text\": \"$message\"}"
}

# Use after Claude review
review_result=$(claude -p "Review code" --output-format text)
send_slack "Code review completed:\n$review_result"
```

### Discord Notifications

```bash
#!/bin/bash
# notify-discord.sh

DISCORD_WEBHOOK="$DISCORD_WEBHOOK_URL"

send_discord() {
  local message="$1"
  curl -X POST "$DISCORD_WEBHOOK" \
    -H "Content-Type: application/json" \
    -d "{\"content\": \"$message\"}"
}
```

### Email Notifications

```bash
#!/bin/bash
# notify-email.sh

send_email() {
  local subject="$1"
  local body="$2"
  local to="$3"

  echo "$body" | mail -s "$subject" "$to"
}

# Use with Claude review
review=$(claude -p "Daily code review" --output-format text)
send_email "Daily Code Review" "$review" "team@example.com"
```

## Best Practices

### Performance

| Practice | Benefit |
|----------|---------|
| Use `--max-turns 1-3` for hooks | Fast feedback |
| Filter files before review | Reduce tokens |
| Debounce file watchers | Avoid duplicate runs |
| Cache Claude installation | Faster startup |

### Reliability

| Practice | Benefit |
|----------|---------|
| Add timeouts to scripts | Prevent hangs |
| Log all automation runs | Debugging |
| Handle API failures gracefully | Don't block workflow |
| Test hooks locally first | Catch issues early |

### Security

| Practice | Benefit |
|----------|---------|
| Never commit API keys | Use environment variables |
| Limit tool permissions | Reduce risk |
| Review automation output | Catch unexpected changes |
| Use read-only for analysis | Safe by default |

## Troubleshooting

| Issue | Solution |
|-------|----------|
| Hook not running | Check permissions (`chmod +x`) |
| Hook too slow | Reduce `--max-turns`, filter files |
| API key not found | Check environment variable export |
| Cron not running | Check cron logs, path issues |
| Webhook not receiving | Check URL, firewall, HTTPS |
