# Vulnerability Prevention Guide

Practical patterns for preventing OWASP Top 10 and common security vulnerabilities.

## OWASP Top 10 (2021)

### A01: Broken Access Control

The most critical web application security risk.

#### Prevention Patterns

```typescript
// 1. Deny by default
function checkAccess(user: User, resource: Resource): boolean {
  // Explicit allow list, not deny list
  const allowedRoles = resource.allowedRoles || [];
  return allowedRoles.includes(user.role);
}

// 2. Enforce ownership
async function getDocument(userId: string, docId: string) {
  const doc = await db.documents.findUnique({ where: { id: docId } });

  if (!doc) {
    throw new NotFoundError('Document not found');
  }

  // Always verify ownership
  if (doc.ownerId !== userId && !doc.sharedWith.includes(userId)) {
    throw new ForbiddenError('Access denied');
  }

  return doc;
}

// 3. Validate indirect object references
const DocumentIdSchema = z.string().uuid();

async function handleRequest(req: Request) {
  const docId = DocumentIdSchema.parse(req.params.id);

  // Use validated ID with ownership check
  return getDocument(req.user.id, docId);
}
```

#### IDOR (Insecure Direct Object Reference) Prevention

```typescript
// BAD - Direct ID exposure
GET /api/users/123/profile

// GOOD - Use opaque tokens or verify ownership
GET /api/users/me/profile  // Current user context

// Or validate access
async function getUserProfile(requesterId: string, targetId: string) {
  // Self-access always allowed
  if (requesterId === targetId) {
    return await getFullProfile(targetId);
  }

  // Check if requester can view target
  const canView = await checkViewPermission(requesterId, targetId);
  if (!canView) {
    throw new ForbiddenError('Cannot view this profile');
  }

  return await getPublicProfile(targetId);
}
```

### A02: Cryptographic Failures

#### Secure Password Storage

```typescript
import { hash, verify } from '@node-rs/argon2';

// Argon2id is the recommended algorithm
const ARGON2_OPTIONS = {
  memoryCost: 65536,    // 64 MB
  timeCost: 3,          // 3 iterations
  parallelism: 4,       // 4 parallel threads
  hashLength: 32,       // 32 bytes output
};

async function hashPassword(password: string): Promise<string> {
  return await hash(password, ARGON2_OPTIONS);
}

async function verifyPassword(hash: string, password: string): Promise<boolean> {
  try {
    return await verify(hash, password);
  } catch {
    return false;
  }
}
```

#### Encryption at Rest

```typescript
import { createCipheriv, createDecipheriv, randomBytes } from 'crypto';

const ALGORITHM = 'aes-256-gcm';
const KEY = Buffer.from(process.env.ENCRYPTION_KEY!, 'base64');

function encrypt(plaintext: string): { ciphertext: string; iv: string; tag: string } {
  const iv = randomBytes(16);
  const cipher = createCipheriv(ALGORITHM, KEY, iv);

  let ciphertext = cipher.update(plaintext, 'utf8', 'base64');
  ciphertext += cipher.final('base64');

  return {
    ciphertext,
    iv: iv.toString('base64'),
    tag: cipher.getAuthTag().toString('base64'),
  };
}

function decrypt(ciphertext: string, iv: string, tag: string): string {
  const decipher = createDecipheriv(
    ALGORITHM,
    KEY,
    Buffer.from(iv, 'base64')
  );
  decipher.setAuthTag(Buffer.from(tag, 'base64'));

  let plaintext = decipher.update(ciphertext, 'base64', 'utf8');
  plaintext += decipher.final('utf8');

  return plaintext;
}
```

#### TLS Configuration

```typescript
import { createServer } from 'https';
import { readFileSync } from 'fs';

const server = createServer({
  key: readFileSync('server.key'),
  cert: readFileSync('server.crt'),
  // Modern TLS configuration
  minVersion: 'TLSv1.2',
  ciphers: [
    'ECDHE-ECDSA-AES128-GCM-SHA256',
    'ECDHE-RSA-AES128-GCM-SHA256',
    'ECDHE-ECDSA-AES256-GCM-SHA384',
    'ECDHE-RSA-AES256-GCM-SHA384',
  ].join(':'),
});
```

### A03: Injection

#### SQL Injection Prevention

```typescript
// Using Drizzle ORM (safe by default)
import { eq, and, like } from 'drizzle-orm';

async function searchUsers(query: string, orgId: string) {
  return await db.select()
    .from(users)
    .where(
      and(
        eq(users.organizationId, orgId),
        like(users.name, `%${query}%`)  // Drizzle escapes this
      )
    );
}

// Raw SQL with parameterization
async function rawQuery(userId: string) {
  return await db.execute(
    sql`SELECT * FROM users WHERE id = ${userId}`
  );
}

// NEVER do this
const badQuery = `SELECT * FROM users WHERE id = '${userId}'`;  // SQL INJECTION!
```

#### NoSQL Injection Prevention

```typescript
// MongoDB - BAD
const user = await User.findOne({
  username: req.body.username,
  password: req.body.password,  // Could be { $ne: null }
});

// MongoDB - GOOD
const username = String(req.body.username);
const password = String(req.body.password);

const user = await User.findOne({ username });
if (user && await verifyPassword(user.passwordHash, password)) {
  // Authenticated
}
```

#### Command Injection Prevention

```typescript
import { execFile } from 'child_process';
import { promisify } from 'util';

const execFileAsync = promisify(execFile);

// BAD - Shell injection possible
exec(`convert ${filename} output.png`);

// GOOD - No shell interpretation
await execFileAsync('convert', [filename, 'output.png']);

// BETTER - Validate and sanitize
function sanitizeFilename(filename: string): string {
  // Remove path traversal
  const basename = path.basename(filename);
  // Allow only safe characters
  return basename.replace(/[^a-zA-Z0-9._-]/g, '');
}

const safeFilename = sanitizeFilename(userInput);
await execFileAsync('convert', [safeFilename, 'output.png']);
```

#### LDAP Injection Prevention

```typescript
// Escape special LDAP characters
function escapeLdapFilter(input: string): string {
  return input
    .replace(/\\/g, '\\5c')
    .replace(/\*/g, '\\2a')
    .replace(/\(/g, '\\28')
    .replace(/\)/g, '\\29')
    .replace(/\x00/g, '\\00');
}

const filter = `(uid=${escapeLdapFilter(username)})`;
```

### A04: Insecure Design

#### Threat Modeling Questions

Before implementing a feature, ask:

1. What data does this handle? (PII, credentials, financial)
2. Who should have access? (roles, ownership)
3. What could an attacker do? (injection, bypass, escalation)
4. What's the blast radius? (scope of compromise)

#### Secure Design Patterns

```typescript
// Rate limiting for sensitive operations
const sensitiveOpsLimiter = rateLimit({
  windowMs: 60 * 1000,
  max: 3,
  keyGenerator: (req) => `${req.ip}-${req.user?.id}`,
});

// Require re-authentication for sensitive actions
async function changePassword(req: Request) {
  // Require current password even when logged in
  const { currentPassword, newPassword } = req.body;

  const user = await getUser(req.user.id);
  if (!await verifyPassword(user.passwordHash, currentPassword)) {
    throw new UnauthorizedError('Current password incorrect');
  }

  await updatePassword(user.id, newPassword);
}

// Business logic validation
async function transferFunds(fromAccount: string, toAccount: string, amount: number) {
  // Validate business rules, not just data types
  if (amount <= 0) {
    throw new ValidationError('Amount must be positive');
  }

  const from = await getAccount(fromAccount);
  if (from.balance < amount) {
    throw new ValidationError('Insufficient funds');
  }

  if (from.dailyTransferred + amount > from.dailyLimit) {
    throw new ValidationError('Daily transfer limit exceeded');
  }

  // Proceed with transfer
}
```

### A05: Security Misconfiguration

#### Secure Default Configuration

```typescript
// Security headers middleware
function securityHeaders(req: Request, res: Response, next: NextFunction) {
  // Prevent MIME type sniffing
  res.setHeader('X-Content-Type-Options', 'nosniff');

  // Prevent clickjacking
  res.setHeader('X-Frame-Options', 'DENY');

  // Enable XSS filter
  res.setHeader('X-XSS-Protection', '1; mode=block');

  // HTTPS only
  res.setHeader('Strict-Transport-Security', 'max-age=31536000; includeSubDomains');

  // CSP
  res.setHeader('Content-Security-Policy', [
    "default-src 'self'",
    "script-src 'self'",
    "style-src 'self' 'unsafe-inline'",
    "img-src 'self' data: https:",
    "font-src 'self'",
    "connect-src 'self'",
    "frame-ancestors 'none'",
  ].join('; '));

  // Referrer policy
  res.setHeader('Referrer-Policy', 'strict-origin-when-cross-origin');

  next();
}
```

#### Production Configuration Checklist

```typescript
// config/production.ts
export const productionConfig = {
  // Disable debug mode
  debug: false,

  // Disable stack traces in errors
  showStackTrace: false,

  // Use secure cookies
  cookies: {
    secure: true,
    httpOnly: true,
    sameSite: 'strict',
  },

  // Disable unnecessary endpoints
  enableSwagger: false,
  enableGraphiQL: false,

  // Logging without sensitive data
  logging: {
    level: 'info',
    redactFields: ['password', 'token', 'secret', 'apiKey'],
  },
};
```

### A06: Vulnerable and Outdated Components

#### Dependency Scanning

```bash
# Regular vulnerability checks
bun audit

# Generate SBOM
npx @cyclonedx/cdxgen -o sbom.json

# Check with Snyk
npx snyk test

# Monitor dependencies
npx snyk monitor
```

#### Automated Updates

```yaml
# .github/dependabot.yml
version: 2
updates:
  - package-ecosystem: "npm"
    directory: "/"
    schedule:
      interval: "weekly"
    open-pull-requests-limit: 10
    groups:
      security:
        patterns:
          - "*"
        update-types:
          - "security"
```

### A07: Identification and Authentication Failures

#### Secure Authentication Implementation

```typescript
import { hash, verify } from '@node-rs/argon2';
import { randomBytes } from 'crypto';

// Account lockout after failed attempts
const MAX_ATTEMPTS = 5;
const LOCKOUT_DURATION = 15 * 60 * 1000; // 15 minutes

async function attemptLogin(email: string, password: string, ip: string) {
  const user = await getUserByEmail(email);

  // Check lockout (constant time to prevent user enumeration)
  if (user?.lockedUntil && user.lockedUntil > new Date()) {
    await delay(randomInt(100, 500)); // Random delay
    throw new UnauthorizedError('Account temporarily locked');
  }

  // Verify password (constant time)
  const isValid = user ? await verify(user.passwordHash, password) : false;

  if (!isValid) {
    if (user) {
      await incrementFailedAttempts(user.id);
    }
    await delay(randomInt(100, 500)); // Prevent timing attacks
    throw new UnauthorizedError('Invalid credentials');
  }

  // Reset failed attempts on success
  await resetFailedAttempts(user!.id);

  // Create session
  const sessionToken = randomBytes(32).toString('base64url');
  await createSession(user!.id, sessionToken, ip);

  return { user: user!, sessionToken };
}

// Constant-time comparison for sensitive values
function safeCompare(a: string, b: string): boolean {
  if (a.length !== b.length) {
    return false;
  }

  let result = 0;
  for (let i = 0; i < a.length; i++) {
    result |= a.charCodeAt(i) ^ b.charCodeAt(i);
  }
  return result === 0;
}
```

#### Multi-Factor Authentication

```typescript
import { authenticator } from 'otplib';

// Generate TOTP secret
function generateTotpSecret(): { secret: string; uri: string } {
  const secret = authenticator.generateSecret();
  const uri = authenticator.keyuri(user.email, 'YourApp', secret);
  return { secret, uri };
}

// Verify TOTP
function verifyTotp(token: string, secret: string): boolean {
  return authenticator.verify({ token, secret });
}

// Backup codes
function generateBackupCodes(): string[] {
  const codes: string[] = [];
  for (let i = 0; i < 10; i++) {
    codes.push(randomBytes(4).toString('hex'));
  }
  return codes;
}
```

### A08: Software and Data Integrity Failures

#### Subresource Integrity

```html
<!-- Verify external scripts -->
<script
  src="https://cdn.example.com/lib.js"
  integrity="sha384-abc123..."
  crossorigin="anonymous">
</script>
```

#### Signed Updates

```typescript
import { verify } from 'crypto';

async function verifyUpdate(data: Buffer, signature: Buffer): Promise<boolean> {
  const publicKey = await getPublicKey();
  return verify(
    'sha256',
    data,
    {
      key: publicKey,
      padding: crypto.constants.RSA_PKCS1_PSS_PADDING,
    },
    signature
  );
}
```

### A09: Security Logging and Monitoring Failures

#### Comprehensive Security Logging

```typescript
interface SecurityEvent {
  timestamp: Date;
  eventType: 'auth' | 'access' | 'modification' | 'error';
  severity: 'info' | 'warning' | 'critical';
  userId?: string;
  ip: string;
  action: string;
  resource?: string;
  outcome: 'success' | 'failure';
  details?: Record<string, unknown>;
}

function logSecurityEvent(event: SecurityEvent): void {
  // Never log sensitive data
  const sanitized = {
    ...event,
    details: redactSensitive(event.details),
  };

  logger.info('security_event', sanitized);

  // Alert on critical events
  if (event.severity === 'critical') {
    alertSecurityTeam(sanitized);
  }
}

// Log authentication events
logSecurityEvent({
  timestamp: new Date(),
  eventType: 'auth',
  severity: 'info',
  userId: user.id,
  ip: req.ip,
  action: 'login',
  outcome: 'success',
});

// Log access failures
logSecurityEvent({
  timestamp: new Date(),
  eventType: 'access',
  severity: 'warning',
  userId: req.user?.id,
  ip: req.ip,
  action: 'access_denied',
  resource: req.path,
  outcome: 'failure',
  details: { reason: 'insufficient_permissions' },
});
```

### A10: Server-Side Request Forgery (SSRF)

#### SSRF Prevention

```typescript
import { URL } from 'url';
import { isPrivate } from 'ip';

const ALLOWED_PROTOCOLS = ['http:', 'https:'];
const BLOCKED_HOSTS = ['localhost', '127.0.0.1', '0.0.0.0', '::1'];

async function safeFetch(urlString: string): Promise<Response> {
  const url = new URL(urlString);

  // Validate protocol
  if (!ALLOWED_PROTOCOLS.includes(url.protocol)) {
    throw new ValidationError('Invalid protocol');
  }

  // Block localhost and private IPs
  if (BLOCKED_HOSTS.includes(url.hostname)) {
    throw new ValidationError('Blocked host');
  }

  // Resolve DNS and check if private
  const addresses = await dns.promises.resolve(url.hostname);
  for (const address of addresses) {
    if (isPrivate(address)) {
      throw new ValidationError('Private IP blocked');
    }
  }

  // Fetch with timeout and size limit
  const controller = new AbortController();
  const timeout = setTimeout(() => controller.abort(), 10000);

  try {
    const response = await fetch(url.toString(), {
      signal: controller.signal,
      redirect: 'manual',  // Handle redirects manually to revalidate
    });

    // Check redirect destination
    if (response.status >= 300 && response.status < 400) {
      const location = response.headers.get('location');
      if (location) {
        return safeFetch(location);  // Recursively validate redirect
      }
    }

    return response;
  } finally {
    clearTimeout(timeout);
  }
}
```

## Common Vulnerability Patterns

### Path Traversal

```typescript
// BAD - User controls file path
const file = path.join(uploadDir, req.params.filename);
return fs.readFileSync(file);  // ../../../../etc/passwd

// GOOD - Validate and normalize
function safeFilePath(uploadDir: string, filename: string): string {
  // Remove path separators and parent directory references
  const safeName = path.basename(filename);

  // Construct full path
  const fullPath = path.resolve(uploadDir, safeName);

  // Verify it's still within upload directory
  if (!fullPath.startsWith(path.resolve(uploadDir))) {
    throw new ValidationError('Invalid file path');
  }

  return fullPath;
}
```

### Mass Assignment

```typescript
// BAD - Accept all fields
const user = await User.create(req.body);  // Could include isAdmin: true

// GOOD - Explicitly pick allowed fields
const allowedFields = ['name', 'email', 'avatar'];
const userData = pick(req.body, allowedFields);
const user = await User.create(userData);

// BETTER - Use schema validation
const CreateUserSchema = z.object({
  name: z.string().min(1).max(100),
  email: z.string().email(),
  avatar: z.string().url().optional(),
});

const userData = CreateUserSchema.parse(req.body);
const user = await User.create(userData);
```

### Race Conditions

```typescript
// BAD - Race condition in balance check
async function withdraw(userId: string, amount: number) {
  const account = await getAccount(userId);
  if (account.balance >= amount) {  // Check
    await updateBalance(userId, account.balance - amount);  // Update
  }
}

// GOOD - Use database transaction with lock
async function withdraw(userId: string, amount: number) {
  return await db.transaction(async (tx) => {
    // Lock row for update
    const account = await tx.query(
      'SELECT * FROM accounts WHERE user_id = ? FOR UPDATE',
      [userId]
    );

    if (account.balance < amount) {
      throw new ValidationError('Insufficient funds');
    }

    await tx.query(
      'UPDATE accounts SET balance = balance - ? WHERE user_id = ?',
      [amount, userId]
    );

    return { newBalance: account.balance - amount };
  });
}
```

### XML External Entity (XXE)

```typescript
import { XMLParser } from 'fast-xml-parser';

// Safe XML parsing configuration
const parser = new XMLParser({
  allowBooleanAttributes: true,
  // Disable dangerous features
  processEntities: false,
  htmlEntities: false,
});

const result = parser.parse(xmlString);
```

### Prototype Pollution

```typescript
// BAD - Direct object assignment
function merge(target: any, source: any) {
  for (const key in source) {
    target[key] = source[key];  // Can modify __proto__
  }
}

// GOOD - Check for dangerous keys
function safeMerge(target: Record<string, unknown>, source: Record<string, unknown>) {
  const BLOCKED_KEYS = ['__proto__', 'constructor', 'prototype'];

  for (const key of Object.keys(source)) {
    if (BLOCKED_KEYS.includes(key)) {
      continue;
    }
    target[key] = source[key];
  }

  return target;
}

// BETTER - Use Object.create(null) for untrusted data
const safeObject = Object.create(null);
Object.assign(safeObject, trustedData);
```

## Security Testing Prompts

Use these prompts with Claude Code:

```
"Test this endpoint for SQL injection vulnerabilities using common payloads"

"Check if this authentication flow is vulnerable to timing attacks"

"Review this file upload handler for path traversal and file type validation issues"

"Analyze this code for potential SSRF vulnerabilities"

"Check if this API is vulnerable to mass assignment attacks"
```
