# Format Configuration Schema

This document describes the JSON schema for format configuration files generated by `extract_format.py`.

## Overview

The format configuration file stores extracted formatting information from a Word document, including styles, numbering, table structures, and header/footer information.

## Schema Structure

```json
{
  "source_document": "string",
  "styles": {},
  "numbering": {},
  "tables": [],
  "headers_footers": {}
}
```

## Fields

### source_document

**Type**: `string`
**Description**: Name of the source document from which the format was extracted.

**Example**:
```json
"source_document": "template.docx"
```

### styles

**Type**: `object`
**Description**: Dictionary of style definitions, keyed by style ID.

Each style object contains:
- `id` (string): Style identifier
- `name` (string): Human-readable style name
- `type` (string): Style type (paragraph, character, table, numbering)
- `fonts` (object): Font properties
  - `ascii` (string): ASCII font name
  - `hAnsi` (string): High ANSI font name
  - `eastAsia` (string): East Asian font name
  - `size` (string): Font size in half-points
- `paragraph` (object): Paragraph properties
  - `alignment` (string): Text alignment (left, center, right, both)
  - `spacing` (object): Line spacing configuration

**Example**:
```json
"styles": {
  "1": {
    "id": "1",
    "name": "Normal",
    "type": "paragraph",
    "fonts": {
      "ascii": "Times New Roman",
      "hAnsi": "Times New Roman",
      "eastAsia": "宋体",
      "size": "24"
    },
    "paragraph": {
      "alignment": "left",
      "spacing": {
        "line": "360",
        "lineRule": "auto"
      }
    }
  },
  "2": {
    "id": "2",
    "name": "heading 1",
    "type": "paragraph",
    "fonts": {
      "ascii": "Times New Roman",
      "size": "32"
    },
    "paragraph": {
      "alignment": "left"
    }
  }
}
```

### numbering

**Type**: `object`
**Description**: Numbering definitions for automatic numbering (1, 1.1, 1.1.1, etc.)

Each numbering entry maps a numbering ID to its abstract numbering definition.

**Example**:
```json
"numbering": {
  "1": {
    "abstractNumId": "0"
  },
  "2": {
    "abstractNumId": "1"
  }
}
```

### tables

**Type**: `array`
**Description**: Array of table structure definitions found in the document.

Each table object contains:
- `index` (number): Zero-based index of the table
- `rows` (number): Number of rows in the table
- `columns` (array): Array of column widths
- `properties` (object): Table properties
  - `width` (object): Table width settings
  - `style` (string): Table style ID

**Example**:
```json
"tables": [
  {
    "index": 0,
    "rows": 5,
    "columns": ["1984", "8076", "40"],
    "properties": {
      "width": {
        "value": "10100",
        "type": "dxa"
      },
      "style": "49"
    }
  }
]
```

### headers_footers

**Type**: `object`
**Description**: Information about headers and footers in the document.

Contains:
- `headers` (array): List of header files
- `footers` (array): List of footer files

**Example**:
```json
"headers_footers": {
  "headers": [
    {
      "file": "word/header1.xml",
      "exists": true
    }
  ],
  "footers": [
    {
      "file": "word/footer1.xml",
      "exists": true
    }
  ]
}
```

## Usage Notes

### Style IDs vs Style Names

Word documents use numeric style IDs internally (e.g., "1", "2") which map to named styles (e.g., "Normal", "heading 1"). The format configuration preserves both for maximum compatibility.

### Width Units

Widths in OOXML use "dxa" units (twentieths of a point). Common conversions:
- 1 inch = 1440 dxa
- 1 cm = 567 dxa
- 1 pt = 20 dxa

### Extracting Additional Properties

The current schema covers the most common formatting properties. For specialized documents, you may need to:

1. Modify `extract_format.py` to extract additional properties
2. Update this schema documentation accordingly
3. Update `generate_document.py` to apply those properties

## Example Complete Configuration

```json
{
  "source_document": "research_task_template.docx",
  "styles": {
    "1": {
      "id": "1",
      "name": "Normal",
      "type": "paragraph",
      "fonts": {
        "ascii": "Times New Roman",
        "hAnsi": "Times New Roman",
        "eastAsia": "宋体",
        "size": "24"
      },
      "paragraph": {
        "alignment": "left"
      }
    }
  },
  "numbering": {
    "1": {
      "abstractNumId": "0"
    }
  },
  "tables": [
    {
      "index": 0,
      "rows": 8,
      "columns": ["2000", "8000"],
      "properties": {
        "width": {
          "value": "10000",
          "type": "dxa"
        },
        "style": "GridTable"
      }
    }
  ],
  "headers_footers": {
    "headers": [
      {"file": "word/header1.xml", "exists": true}
    ],
    "footers": [
      {"file": "word/footer1.xml", "exists": true}
    ]
  }
}
```
