#!/bin/bash
# Create init.sh template for dev server startup
# From Anthropic Effective Harnesses: "init.sh script for development server startup"
# Usage: ./create-init-script.sh [project-type]

set -e

PROJECT_TYPE="${1:-auto}"

# Auto-detect project type if not specified
if [ "$PROJECT_TYPE" = "auto" ]; then
    if [ -f "pyproject.toml" ] || [ -f "requirements.txt" ]; then
        if grep -q "fastapi" pyproject.toml requirements.txt 2>/dev/null; then
            PROJECT_TYPE="fastapi"
        elif grep -q "django" pyproject.toml requirements.txt 2>/dev/null; then
            PROJECT_TYPE="django"
        elif grep -q "flask" pyproject.toml requirements.txt 2>/dev/null; then
            PROJECT_TYPE="flask"
        else
            PROJECT_TYPE="python"
        fi
    elif [ -f "package.json" ]; then
        if grep -q '"next"' package.json; then
            PROJECT_TYPE="nextjs"
        elif grep -q '"express"' package.json; then
            PROJECT_TYPE="express"
        else
            PROJECT_TYPE="node"
        fi
    elif [ -f "Cargo.toml" ]; then
        PROJECT_TYPE="rust"
    elif [ -f "go.mod" ]; then
        PROJECT_TYPE="go"
    else
        PROJECT_TYPE="generic"
    fi
fi

echo "Detected project type: $PROJECT_TYPE"

# Create scripts directory
mkdir -p scripts

# Generate init.sh based on project type
case "$PROJECT_TYPE" in
    fastapi)
        cat > scripts/init.sh << 'EOF'
#!/bin/bash
# FastAPI Development Server
# Generated by agent-harness

set -e

# Check for virtual environment
if [ ! -d "venv" ]; then
    echo "Creating virtual environment..."
    python -m venv venv
fi

# Activate venv
source venv/bin/activate

# Install dependencies
pip install -r requirements.txt -q

# Run database migrations (if alembic configured)
if [ -f "alembic.ini" ]; then
    echo "Running migrations..."
    alembic upgrade head
fi

# Start dev server
echo "Starting FastAPI server on http://localhost:8000"
uvicorn main:app --reload --host 0.0.0.0 --port 8000
EOF
        PORT=8000
        HEALTH_CHECK="curl -sf http://localhost:8000/health || curl -sf http://localhost:8000/docs"
        ;;

    django)
        cat > scripts/init.sh << 'EOF'
#!/bin/bash
# Django Development Server
# Generated by agent-harness

set -e

# Check for virtual environment
if [ ! -d "venv" ]; then
    echo "Creating virtual environment..."
    python -m venv venv
fi

# Activate venv
source venv/bin/activate

# Install dependencies
pip install -r requirements.txt -q

# Run migrations
echo "Running migrations..."
python manage.py migrate

# Start dev server
echo "Starting Django server on http://localhost:8000"
python manage.py runserver 0.0.0.0:8000
EOF
        PORT=8000
        HEALTH_CHECK="curl -sf http://localhost:8000/admin/login/ || curl -sf http://localhost:8000/"
        ;;

    flask)
        cat > scripts/init.sh << 'EOF'
#!/bin/bash
# Flask Development Server
# Generated by agent-harness

set -e

# Check for virtual environment
if [ ! -d "venv" ]; then
    echo "Creating virtual environment..."
    python -m venv venv
fi

# Activate venv
source venv/bin/activate

# Install dependencies
pip install -r requirements.txt -q

# Set Flask environment
export FLASK_APP=${FLASK_APP:-app.py}
export FLASK_ENV=development

# Start dev server
echo "Starting Flask server on http://localhost:5000"
flask run --host=0.0.0.0 --port=5000
EOF
        PORT=5000
        HEALTH_CHECK="curl -sf http://localhost:5000/health || curl -sf http://localhost:5000/"
        ;;

    nextjs)
        cat > scripts/init.sh << 'EOF'
#!/bin/bash
# Next.js Development Server
# Generated by agent-harness

set -e

# Install dependencies
if [ ! -d "node_modules" ]; then
    echo "Installing dependencies..."
    npm install
fi

# Start dev server
echo "Starting Next.js server on http://localhost:3000"
npm run dev
EOF
        PORT=3000
        HEALTH_CHECK="curl -sf http://localhost:3000/"
        ;;

    express)
        cat > scripts/init.sh << 'EOF'
#!/bin/bash
# Express Development Server
# Generated by agent-harness

set -e

# Install dependencies
if [ ! -d "node_modules" ]; then
    echo "Installing dependencies..."
    npm install
fi

# Start dev server
echo "Starting Express server on http://localhost:3000"
npm run dev || node server.js || node index.js
EOF
        PORT=3000
        HEALTH_CHECK="curl -sf http://localhost:3000/health || curl -sf http://localhost:3000/"
        ;;

    node)
        cat > scripts/init.sh << 'EOF'
#!/bin/bash
# Node.js Development Server
# Generated by agent-harness

set -e

# Install dependencies
if [ ! -d "node_modules" ]; then
    echo "Installing dependencies..."
    npm install
fi

# Start dev server
echo "Starting Node.js server..."
npm start || node index.js
EOF
        PORT=3000
        HEALTH_CHECK="curl -sf http://localhost:3000/"
        ;;

    python)
        cat > scripts/init.sh << 'EOF'
#!/bin/bash
# Python Application
# Generated by agent-harness

set -e

# Check for virtual environment
if [ ! -d "venv" ]; then
    echo "Creating virtual environment..."
    python -m venv venv
fi

# Activate venv
source venv/bin/activate

# Install dependencies
if [ -f "requirements.txt" ]; then
    pip install -r requirements.txt -q
elif [ -f "pyproject.toml" ]; then
    pip install -e . -q
fi

echo "Python environment ready"
echo "Run your application manually"
EOF
        PORT=""
        HEALTH_CHECK=""
        ;;

    *)
        cat > scripts/init.sh << 'EOF'
#!/bin/bash
# Generic Development Setup
# Generated by agent-harness
# Customize this script for your project

set -e

echo "=== Development Environment Setup ==="

# Add your setup commands here:
# - Install dependencies
# - Run migrations
# - Start services

echo "Setup complete. Start your dev server manually."
EOF
        PORT=""
        HEALTH_CHECK=""
        ;;
esac

# Make executable
chmod +x scripts/init.sh

echo "Created scripts/init.sh for $PROJECT_TYPE"

# Update or create project.json
mkdir -p .claude/config

if [ -f ".claude/config/project.json" ]; then
    # Update existing config
    jq --arg type "$PROJECT_TYPE" \
       --arg port "$PORT" \
       --arg health "$HEALTH_CHECK" \
       '. + {project_type: $type, init_script: "./scripts/init.sh"} |
        if $port != "" then . + {dev_server_port: ($port | tonumber)} else . end |
        if $health != "" then . + {health_check: $health} else . end' \
       .claude/config/project.json > /tmp/project.json.tmp && \
       mv /tmp/project.json.tmp .claude/config/project.json
    echo "Updated .claude/config/project.json"
else
    # Create new config
    cat > .claude/config/project.json << EOF
{
  "project_type": "$PROJECT_TYPE",
  "init_script": "./scripts/init.sh",
  "dev_server_port": ${PORT:-null},
  "health_check": ${HEALTH_CHECK:+"\"$HEALTH_CHECK\""}${HEALTH_CHECK:-null},
  "test_command": "pytest -q --tb=short",
  "required_env": [],
  "required_services": []
}
EOF
    echo "Created .claude/config/project.json"
fi

# Output summary
cat << EOF

=== Init Script Created ===
File: scripts/init.sh
Type: $PROJECT_TYPE
Port: ${PORT:-"(not configured)"}
Health: ${HEALTH_CHECK:-"(not configured)"}

Run: ./scripts/init.sh
EOF
