# 仕様書設計の背景

## Claudeの時代における仕様書

Claudeはコードを読んで仕様を抽出できる。だから「コードから読み取れる情報」を仕様書に書く価値はない。

価値があるのは「コードから読み取れない情報」だけ。

- なぜこの設計にしたか
- 採用しなかった選択肢
- トレードオフの判断

## レイヤーに沿った構造

仕様書はアーキテクチャのレイヤーに対応させる。

| レイヤー | ディレクトリ | 内容 | 必須 |
|---------|-------------|------|------|
| インターフェース層 | interface/ | 機能、画面、導線 | ○ |
| ドメイン層 | domain/ | 関係、集約、不変条件 | ○ |
| アプリケーション層 | application/ | 権限、ユースケース、バックグラウンド処理 | ○ |
| インフラストラクチャ層 | infrastructure/ | DB設計、外部連携 | ○ |

### なぜ分けるか

- インターフェース層 → コンポーネント、ルートで実装
- ドメイン層の不変条件 → schema.ts で実装
- アプリケーション層の権限 → API routes で実装
- インフラストラクチャ層 → DB、外部サービス接続で実装

実装場所が異なるので、ドキュメントも分ける。

## ドメインモデルの体系化

```
ドメインモデル
  ├── 用語集（ユビキタス言語）
  ├── アクター（ロール）
  ├── 関係図
  ├── 集約 + 不変条件
  └── ...
```

| 内容 | ファイル | 理由 |
|------|---------|------|
| 用語集 | glossary.md | コードは英語、ビジネスは日本語。対応表が必要 |
| アクター | domain/actors.md | 固定ロール（ユースケースの主語） |
| 概念の関係 | domain/relationships.md | テーマごとに図示。全体を一度に見せるとごちゃごちゃする |
| 集約ルート + 不変条件 + 状態遷移 | domain/models/*.md | 1モデル1ファイルで管理 |

エンティティの属性や関係はコードから読めるので書かない。

## ユースケースの体系化

### 機能とユースケースの関係

機能とユースケースは同じものの別視点。どちらか一方で網羅できる。

- 機能 → システム中心（何ができるか）
- ユースケース → ユーザー中心（何を達成できるか）

ユーザー中心のプロダクトはユースケースで整理する。

### 分類ルール

ユースケースは「操作対象（集約）」で分類する。

「〇〇を△△する」の「〇〇」で決まる。

| ユースケース | 操作対象 | ファイル |
|-------------|---------|---------|
| ユーザーを登録する | User | user.md |
| 招待を作成する | Invite | invite.md |
| 投稿を作成する | Post | post.md |

### 複数機能にまたがるユースケース

workflows.md に書く。

例: 月次表彰（フラグ + フィルタ + エクスポート）

### フォーマット

```markdown
# Post

## CreatePost - 投稿を作成する

従業員と管理者が投稿を作成できる。タイプ選択を最初にすることで投稿の目的を明確にする[^1]。

---

[^1]: 2024-01-15 - 自動分類を検討したが精度問題で却下
```

## メタ情報を本文に書かない理由

各ファイルに「このドキュメントは〜を記録する」と書くと:

- 冗長になる
- 読むたびに同じ説明を見る
- 本題に入るまでが遅い

メタ情報は SKILL.md に集約し、各ファイルは本文から始める。

## 関連ファイルへのリンクを書かない理由

他のファイルへのリンク（`→ [xxx.md](path/to/xxx.md)` 等）は禁止。

- メンテナンスコストが上がる
- ファイル追加・削除時に複数箇所を更新する必要がある
- リンク切れが発生しやすい
- Claude は SKILL.md を見れば構成を理解できる

構成は SKILL.md に一元化する。

## 冗長な見出しを使わない理由

「なぜ〇〇か」のような見出しを毎回使うと:

- パターン化して意味が薄れる
- 見出しが長くなる
- 本文で説明すれば十分

代わりに「設計意図」のような汎用的な見出しを使い、本文で自然に説明する。

## 用語集のフォーマット

見出しに「日本語名 - PascalCase」、本文に説明。

```markdown
## ヒヤリ - NearMiss

危なかった経験。
```

テーブルではなく見出しを使う理由:

- 検索しやすい
- 説明を複数行書ける
- 拡張しやすい

コードと完全に一致しなくてもいい。用語集が正で、コードを合わせる。

## 関係図のフォーマット

全体図を先頭に、詳細をテーマごとに配置。

```markdown
## 全体

（全体の関係図）

## 投稿

（投稿に関する関係図と説明）

## エンゲージメント

（エンゲージメントに関する関係図と説明）
```

全体を1つの図にまとめると見づらい。テーマごとに分割して、それぞれに説明を添える。

## モデルのフォーマット

1集約ルート1ファイル。ファイル名にモデル名、本文に集約の構造と不変条件。

```markdown
# Post

Post（ルート）
  ├── Reaction
  └── Comment

Reaction と Comment は Post なしに存在できない。

## 不変条件

**タイプは変更不可**

後から変更できると...
```

「なぜ〇〇が集約ルートか」のような冗長な見出しパターンは使わない。

## ディレクトリ分割の判断

小規模なうちは1ファイルで十分。肥大化したら分割する。

ただし domain/ と application/ は最初から分割した。理由:

- レイヤーが異なる
- 実装場所が異なる
- 責務が明確に分かれる

## 製品パターンの設計

製品の種類によってインターフェース層の構成が変わる。

| パターン | 例 | interface/ に必要なもの |
|---------|------|----------------------|
| Web + API | 一般的な Web アプリ | sitemap.md, api.md, features/, pages/ |
| API のみ | マイクロサービス、BFF | api.md, features/ |
| Web のみ | LP、静的サイト、SPA | sitemap.md, features/, pages/ |

### なぜパターンを分けるか

- API サーバに画面設計は不要
- 静的サイトに API 設計は不要
- 不要なファイルを作ると混乱を招く

### 複数製品の場合

モノレポで複数製品を管理する場合、`products/{product}/` 構造を使う。

- glossary.md は共通（用語は製品をまたいで統一）
- architecture/ は共通（技術選定は組織レベル）
- 各製品の仕様は products/ 配下に分離

## 運用ドキュメントの位置付け

運用（監視、デプロイ、アラート）は infrastructure/operations.md に書く。

### なぜ infrastructure/ か

- 運用はインフラ層の責務
- DB設計や外部連携と同じレイヤー
- 実装は IaC やCI/CD設定で行う

### 何を書くか

具体的な設定値や手順書はコード（Terraform、GitHub Actions等）に書く。operations.md には「なぜそうするか」だけを書く。

- なぜこのメトリクスを監視するか
- なぜこの閾値でアラートを出すか
- なぜこのデプロイ戦略を選んだか

## ADR（Architecture Decision Records）

個別の設計判断は decisions/ に記録する。

- 状況、選択肢、決定、理由を書く
- 後から「なぜこうなっているか」を追える
- フォーマットは SKILL.md を参照

日常の技術選定は architecture/overview.md に書く。大きな判断や議論があったものは ADR として残す。

## 文章と脚注

### なぜ脚注で変更履歴を残すか

本文は「現在の決定事項」、脚注は「その経緯」という役割分担。

- 本文がクリーンに保たれる
- 変更が文脈に紐づく（どの決定がいつ・なぜ変わったか）
- ADRより軽量（大きな判断はADR、小さな判断は脚注）

### なぜ見出しではなく自然な文章か

見出しで区切ると:

- 「なぜ〇〇か」のような冗長なパターンが増える
- 読みにくくなる
- 本質が埋もれる

自然な文章にすることで、決定事項が一目で読める。

### 脚注のフォーマット

```markdown
タイプは後から変更できない[^1]。

---

[^1]: 2024-01-15 - 変更可能にする要望があったが統計の一貫性のため見送り
```

日付と理由をセットで記録する。これにより「いつ・なぜ」が追跡できる。
