---
title: Use Generated Types
impact: MEDIUM
impactDescription: Ensures type safety between server and client
tags: typescript, types, codegen, safety
---

## Use Generated Types

**Impact: MEDIUM (Ensures type safety between server and client)**

Always use the types generated by the SpacetimeDB CLI rather than manually defining them. This ensures your client types exactly match your server schema and prevents runtime type mismatches.

**Incorrect (manually defined types):**

```typescript
// Manually defined types - can drift from server schema!

interface Player {
  id: string; // Wrong! Server uses identity
  name: string;
  score: number;
  isOnline: boolean;
  // Missing lastSeen field that server has!
}

interface Message {
  id: string;
  authorId: string;
  content: string;
  timestamp: number; // Wrong! Server uses bigint (u64)
}

function PlayerCard({ player }: { player: Player }) {
  // Type mismatch - runtime errors waiting to happen
  return <div>{player.name}: {player.score}</div>;
}
```

**Correct (using generated types):**

```typescript
// Import generated types from SpacetimeDB CLI output
import {
  DbConnection,
  Player,
  Message,
  GameState,
  Identity,
} from './generated';

// Types exactly match server schema
function PlayerCard({ player }: { player: Player }) {
  // Full type safety - IDE autocomplete works correctly
  return (
    <div>
      <span>{player.name}</span>
      <span>{player.score.toString()}</span>
      <span>{player.isOnline ? 'Online' : 'Offline'}</span>
      <span>Last seen: {new Date(Number(player.lastSeen)).toLocaleString()}</span>
    </div>
  );
}

// Connection provides typed access to reducers and tables
function useSendMessage(conn: DbConnection) {
  const sendMessage = (channelId: string, content: string) => {
    // TypeScript ensures these arguments match server reducer signature
    return conn.reducers.sendMessage(channelId, content);
  };

  return sendMessage;
}

// Type-safe table access
function useOnlinePlayers(conn: DbConnection): Player[] {
  // Access tables through the connection's db property
  const players = [...conn.db.player.iter()];
  // Filter returns correctly typed array
  return players.filter(p => p.isOnline);
}
```

```bash
# Generate types from your SpacetimeDB module
spacetime generate --lang typescript --out-dir ./src/generated

# Add to package.json scripts for easy regeneration
{
  "scripts": {
    "generate-types": "spacetime generate --lang typescript --out-dir ./src/generated",
    "dev": "npm run generate-types && vite"
  }
}
```

```typescript
// Generated types example (src/generated/index.ts)
// DO NOT EDIT - Auto-generated by SpacetimeDB CLI

export interface Player {
  identity: Identity;
  name: string;
  score: bigint;
  isOnline: boolean;
  lastSeen: bigint;
  createdAt: bigint;
}

export interface Message {
  id: string;
  channelId: string;
  authorId: Identity;
  content: string;
  timestamp: bigint;
}

export interface GameState {
  id: string;
  status: string; // 'waiting' | 'active' | 'finished'
  currentRound: number;
  startedAt: bigint | null;
}

// DbConnection provides typed access to tables and reducers
export interface DbConnection {
  db: {
    player: TableHandle<Player>;
    message: TableHandle<Message>;
    gameState: TableHandle<GameState>;
  };
  reducers: {
    sendMessage: (channelId: string, content: string) => Promise<void>;
    createGame: (name: string, maxPlayers: number) => Promise<void>;
    joinGame: (gameId: string) => Promise<void>;
  };
  subscription: (query: [string, ...any[]]) => Subscription;
}
```

**Regenerate types whenever you:**
- Add or modify tables
- Change reducer signatures
- Update column types
- Add or remove indexes

Reference: [SpacetimeDB TypeScript Client SDK](https://spacetimedb.com/docs/sdks/typescript/)
