---
title: CSP Provider
subtitle: Configures CSP-related behavior for inline tags rendered by Base UI components.
description: A CSP provider component that applies a nonce to inline <style> and <script> tags rendered by Base UI components, and can disable inline <style> elements.
---

# CSP Provider

A CSP provider component that applies a nonce to inline \<style> and \<script> tags rendered by Base UI components, and can disable inline \<style> elements.

## Anatomy

Import the component and wrap it around your app:

```jsx title="Anatomy"
import { CSPProvider } from '@base-ui/react/csp-provider';

// prettier-ignore
<CSPProvider nonce="...">
  {/* Your app or a group of components */}
</CSPProvider>
```

Some Base UI components render inline `<style>` or `<script>` tags for functionality such as removing scrollbars or pre-hydration behavior. Under a strict Content Security Policy (CSP), these tags may be blocked unless they include a matching [nonce](https://developer.mozilla.org/en-US/docs/Web/HTML/Reference/Global_attributes/nonce) attribute.

`CSPProvider` allows configuring this behavior globally for all Base UI components within its tree.

## Supplying a nonce

If you enforce a CSP that blocks inline tags by default, configure your server to:

1. Generate a random nonce per request
2. Include it in your CSP header (via `style-src-elem`/`script-src`)
3. Pass the same nonce into `CSPProvider` during rendering

```ts title="Example"
const nonce = crypto.randomUUID();

// Example CSP header
const csp = [
  `default-src 'self'`,
  `script-src 'self' 'nonce-${nonce}'`,
  `style-src-elem 'self' 'nonce-${nonce}'`,
].join('; ');
```

Then:

```jsx title="Providing the nonce"
import { CSPProvider } from '@base-ui/react/csp-provider';

function App({ nonce }) {
  return <CSPProvider nonce={nonce}>{/* ... */}</CSPProvider>;
}
```

This will ensure that all inline `<style>` and `<script>` tags rendered by Base UI components include the correct nonce attribute, allowing them to function under your CSP.

## Disable inline style elements

You can avoid supplying a `nonce` if you disable inline `<style>` elements entirely and rely on external stylesheets only. The relevant components are `<ScrollArea.Viewport>` and `<Select.Popup>` or `<Select.List>` when `alignItemWithTrigger` is enabled, which inject a style tag to disable native scrollbars.

```html
<style>
  .base-ui-disable-scrollbar {
    scrollbar-width: none;
  }
  .base-ui-disable-scrollbar::-webkit-scrollbar {
    display: none;
  }
</style>
```

Specify `disableStyleElements` to remove these tags:

```jsx title="Disabling style elements"
<CSPProvider disableStyleElements>{/* ... */}</CSPProvider>
```

`<script>` tags across all components are opt-in, so they are not affected by this prop and don't have their own disable flag. A `nonce` is required if any component uses inline scripts.

## Inline style attributes

`CSPProvider` covers inline `<style>` and `<script>` tags rendered as elements, but it does not cover inline style attributes (for example, `<div style="...">`). The `style-src-attr` directive in CSP governs inline style attributes encountered when parsing HTML from server pre-rendered components (it does not affect client-side JavaScript that sets styles).

In CSP, `style-src` applies to both `<style>` elements and `style=""` attributes. If you only want to control `<style>` elements, use `style-src-elem` instead.

If your CSP blocks inline style _attributes_ in addition to _elements_, you have a few options:

1. Relax your CSP by adding `'unsafe-inline'` to the `style-src-attr` directive (or using only `style-src-elem` instead of `style-src`). Style attributes specifically pose a less severe security risk than style elements, but this approach may not be acceptable in high-security environments.
2. Render the affected components only on the client, so that no inline styles are present in the initial HTML.
3. Manually unset inline styles and specify them in your CSS instead. Any component can have its inline styles unset, such as `<ScrollArea.Viewport style={{ overflow: undefined  }}>`. Note that you'll need to ensure you vet upgrades for any new inline styles added by Base UI components.

## API reference

Provides a default Content Security Policy (CSP) configuration for Base UI components that
require inline `<style>` or `<script>` tags.

**CSPProvider Props:**

| Prop                 | Type        | Default | Description                                                                                                                                                                      |
| :------------------- | :---------- | :------ | :------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- |
| disableStyleElements | `boolean`   | `false` | Whether inline `<style>` elements created by Base UI components should not be rendered. Instead, components must specify the CSS styles via custom class names or other methods. |
| nonce                | `string`    | -       | The nonce value to apply to inline `<style>` and `<script>` tags.                                                                                                                |
| children             | `ReactNode` | -       | -                                                                                                                                                                                |
