# Bun S3 API

Native bindings for S3-compatible object storage (AWS S3, Cloudflare R2, DigitalOcean Spaces, MinIO, Google Cloud Storage, Supabase).

## Core Concepts

- `Bun.s3` — global singleton using environment variables
- `Bun.S3Client` — explicit credentials client
- `S3File` — lazy reference extending `Blob`
- Zero network requests until you call a method that needs one

## Environment Variables

```bash
# Primary (Bun-specific)
S3_ACCESS_KEY_ID=...
S3_SECRET_ACCESS_KEY=...
S3_REGION=us-east-1
S3_BUCKET=my-bucket
S3_ENDPOINT=https://s3.us-east-1.amazonaws.com

# Fallback (AWS-style)
AWS_ACCESS_KEY_ID=...
AWS_SECRET_ACCESS_KEY=...
AWS_REGION=...
```

## Basic Usage

```ts
import { s3, S3Client } from "bun";

// Using global singleton (reads env vars)
const file = s3.file("path/to/file.json");

// Using explicit client
const client = new S3Client({
  accessKeyId: "...",
  secretAccessKey: "...",
  bucket: "my-bucket",
  endpoint: "https://s3.us-east-1.amazonaws.com",
});
const file = client.file("path/to/file.json");
```

## Reading Files

```ts
const file = s3.file("data.json");

// All methods return Promise
const text = await file.text();
const json = await file.json();
const buffer = await file.arrayBuffer();
const bytes = await file.bytes();

// Partial read (HTTP Range header)
const partial = await file.slice(0, 1024).text();

// Stream
for await (const chunk of file.stream()) {
  console.log(chunk);
}
```

## Writing Files

```ts
const file = s3.file("output.json");

// Simple write
await file.write("Hello World!");
await file.write(Buffer.from("binary data"));
await file.write(new Response("from response"));

// With options
await file.write(JSON.stringify(data), {
  type: "application/json",
});

await file.write(compressedData, {
  type: "application/json",
  contentEncoding: "gzip",
});

await file.write(pdfData, {
  type: "application/pdf",
  contentDisposition: 'attachment; filename="report.pdf"',
});

// Using Bun.write
await Bun.write(s3.file("output.txt"), "Hello World!");
```

## Streaming Large Files

```ts
const file = s3.file("large-file.bin");

const writer = file.writer({
  type: "application/octet-stream",
  retry: 3,              // Retry on network errors
  queueSize: 10,         // Concurrent uploads
  partSize: 5 * 1024 * 1024, // 5MB chunks
});

for (const chunk of chunks) {
  writer.write(chunk);
  await writer.flush();
}
await writer.end();
```

## Presigned URLs

```ts
// Download URL (default: GET, 24 hours)
const downloadUrl = s3.presign("file.txt");

// Upload URL
const uploadUrl = s3.presign("file.json", {
  method: "PUT",
  expiresIn: 3600,       // 1 hour
  type: "application/json",
  acl: "public-read",
});

// Force download
const forceDownload = s3.presign("report.pdf", {
  contentDisposition: 'attachment; filename="quarterly-report.pdf"',
});
```

### ACL Options

| ACL                           | Description               |
| ----------------------------- | ------------------------- |
| `"private"`                   | Bucket owner only         |
| `"public-read"`               | Public readable           |
| `"public-read-write"`         | Public read/write         |
| `"authenticated-read"`        | Authenticated users       |
| `"bucket-owner-read"`         | Bucket owner readable     |
| `"bucket-owner-full-control"` | Bucket owner full control |

## File Operations

```ts
// Check existence
const exists = await file.exists();

// Get metadata
const stat = await file.stat();
// { etag, lastModified, size, type }

// Get size only
const bytes = await S3Client.size("file.txt", credentials);

// Delete
await file.delete();
// or
await file.unlink();
```

## Listing Objects

```ts
// List up to 1000 objects
const result = await S3Client.list(null, credentials);

// With prefix/pagination
const uploads = await S3Client.list({
  prefix: "uploads/",
  maxKeys: 500,
  fetchOwner: true,
}, credentials);

// Continue if truncated
if (uploads.isTruncated) {
  const more = await S3Client.list({
    prefix: "uploads/",
    startAfter: uploads.contents.at(-1).key,
  }, credentials);
}
```

## S3-Compatible Services

### AWS S3

```ts
const s3 = new S3Client({
  accessKeyId: "...",
  secretAccessKey: "...",
  bucket: "my-bucket",
  region: "us-east-1",
});
```

### Cloudflare R2

```ts
const r2 = new S3Client({
  accessKeyId: "...",
  secretAccessKey: "...",
  bucket: "my-bucket",
  endpoint: "https://<account-id>.r2.cloudflarestorage.com",
});
```

### Google Cloud Storage

```ts
const gcs = new S3Client({
  accessKeyId: "...",
  secretAccessKey: "...",
  bucket: "my-bucket",
  endpoint: "https://storage.googleapis.com",
});
```

### DigitalOcean Spaces

```ts
const spaces = new S3Client({
  accessKeyId: "...",
  secretAccessKey: "...",
  bucket: "my-bucket",
  endpoint: "https://nyc3.digitaloceanspaces.com",
});
```

### MinIO

```ts
const minio = new S3Client({
  accessKeyId: "...",
  secretAccessKey: "...",
  bucket: "my-bucket",
  endpoint: "http://localhost:9000",
});
```

### Supabase

```ts
const supabase = new S3Client({
  accessKeyId: "...",
  secretAccessKey: "...",
  bucket: "my-bucket",
  region: "us-west-1",
  endpoint: "https://<account-id>.supabase.co/storage/v1/s3/storage",
});
```

### Virtual Hosted-Style

```ts
const s3 = new S3Client({
  accessKeyId: "...",
  secretAccessKey: "...",
  bucket: "my-bucket",
  virtualHostedStyle: true,
  // Endpoint auto-inferred from region+bucket
});
```

## s3:// Protocol

```ts
// Works with fetch
const response = await fetch("s3://my-bucket/file.txt");

// Works with Bun.file
const file = Bun.file("s3://my-bucket/file.txt");

// With credentials
const response = await fetch("s3://my-bucket/file.txt", {
  s3: {
    accessKeyId: "...",
    secretAccessKey: "...",
  },
  headers: {
    range: "bytes=0-1023",
  },
});
```

## Quick Redirect to Presigned URL

```ts
// Returns 302 redirect to presigned URL
// Saves bandwidth by not downloading through your server
const response = new Response(s3.file("large-file.zip"));
```

## Error Handling

```ts
try {
  await file.text();
} catch (error) {
  // Bun errors
  if (error.code === "ERR_S3_MISSING_CREDENTIALS") { ... }
  if (error.code === "ERR_S3_INVALID_PATH") { ... }

  // S3 service errors
  if (error.name === "S3Error") { ... }
}
```

### Error Codes

| Code                           | Description              |
| ------------------------------ | ------------------------ |
| `ERR_S3_MISSING_CREDENTIALS`   | Credentials not provided |
| `ERR_S3_INVALID_METHOD`        | Invalid HTTP method      |
| `ERR_S3_INVALID_PATH`          | Invalid file path        |
| `ERR_S3_INVALID_ENDPOINT`      | Invalid endpoint URL     |
| `ERR_S3_INVALID_SIGNATURE`     | Signature mismatch       |
| `ERR_S3_INVALID_SESSION_TOKEN` | Invalid session token    |

## Practical Patterns

### File Upload API

```ts
Bun.serve({
  async fetch(req) {
    if (req.method === "POST") {
      const file = s3.file(`uploads/${Date.now()}`);
      await file.write(req);
      return new Response(JSON.stringify({
        url: file.presign({ expiresIn: 3600 }),
      }));
    }
    return new Response("Upload endpoint", { status: 405 });
  },
});
```

### Direct User Upload

```ts
// Generate presigned PUT URL for client
const uploadUrl = s3.presign(`user-uploads/${userId}/${filename}`, {
  method: "PUT",
  expiresIn: 300,
  type: contentType,
});

// Client uploads directly to S3
// Your server never handles the file data
```

### Download Proxy with Redirect

```ts
Bun.serve({
  fetch(req) {
    const url = new URL(req.url);
    const key = url.pathname.slice(1);

    // Redirect to presigned URL
    return new Response(s3.file(key));
  },
});
```
