# i18n Provider

Refine supports internationalization through a pluggable i18nProvider interface.

## Interface

```ts
interface I18nProvider {
  translate: (key: string, options?: any, defaultMessage?: string) => string;
  changeLocale: (locale: string, options?: any) => Promise<any>;
  getLocale: () => string;
}
```

## Setup with react-i18next

```bash
npm install react-i18next i18next i18next-http-backend i18next-browser-languagedetector
```

```tsx
// src/i18n.ts
import i18n from "i18next";
import { initReactI18next } from "react-i18next";
import Backend from "i18next-http-backend";
import LanguageDetector from "i18next-browser-languagedetector";

i18n
  .use(Backend)
  .use(LanguageDetector)
  .use(initReactI18next)
  .init({
    supportedLngs: ["en", "ru", "de"],
    fallbackLng: "en",
    interpolation: { escapeValue: false },
  });

export default i18n;
```

## Translation Files

```json
// public/locales/en/common.json
{
  "pages.posts.list": "Posts",
  "buttons": { "create": "Create", "save": "Save", "delete": "Delete" },
  "notifications": { "success": "Successful", "error": "Error" },
  "warnWhenUnsavedChanges": "You have unsaved changes."
}
```

## i18nProvider Implementation

```tsx
import { I18nProvider } from "@refinedev/core";
import { useTranslation } from "react-i18next";

export const useI18nProvider = (): I18nProvider => {
  const { t, i18n } = useTranslation();
  return {
    translate: (key, options, defaultMessage) => t(key, { defaultValue: defaultMessage, ...options }),
    changeLocale: (locale) => i18n.changeLanguage(locale),
    getLocale: () => i18n.language,
  };
};

// App.tsx
import "./i18n";
const i18nProvider = useI18nProvider();
<Refine i18nProvider={i18nProvider} />
```

## Hooks

```tsx
import { useTranslation, useGetLocale, useSetLocale } from "@refinedev/core";

const { translate, changeLocale, getLocale } = useTranslation();

translate("pages.posts.list");                    // Simple key
translate("greeting", { name: "John" });          // Interpolation: "Hello, {{name}}!"
translate("missing.key", {}, "Default Text");     // Fallback

const locale = useGetLocale();    // () => "en"
const setLocale = useSetLocale(); // (locale) => Promise
```

## Locale Switcher

```tsx
function LocaleSwitcher() {
  const locale = useGetLocale();
  const setLocale = useSetLocale();

  return (
    <select value={locale()} onChange={(e) => setLocale(e.target.value)}>
      <option value="en">English</option>
      <option value="ru">Русский</option>
    </select>
  );
}
```

## Refine Component Keys

Refine Mantine components auto-translate these keys:

```json
{
  "buttons": { "create": "Create", "save": "Save", "delete": "Delete", "edit": "Edit", "cancel": "Cancel" },
  "actions": { "list": "List", "create": "Create", "edit": "Edit", "show": "Show" },
  "notifications": { "success": "Successful", "error": "Error", "undoable": "You have {{seconds}} seconds to undo" },
  "table": { "actions": "Actions" }
}
```

## Resource Labels

```tsx
<Refine
  resources={[{
    name: "posts",
    list: "/posts",
    meta: { label: "pages.posts.title" }, // Translation key
  }]}
/>
```

Auto-pattern: `${resource.name}.${resource.name}` → `posts.posts`

## Date Formatting with Locale

```tsx
import { useGetLocale } from "@refinedev/core";
import { format } from "date-fns";
import { enUS, ru, de } from "date-fns/locale";

const locales = { en: enUS, ru, de };

function FormattedDate({ date }: { date: Date }) {
  const locale = useGetLocale()() as keyof typeof locales;
  return <span>{format(date, "PPP", { locale: locales[locale] })}</span>;
}
```
