# Debugging and Profiling PyTorch

## Table of Contents

- [Common Issues](#common-issues)
- [Gradient Debugging](#gradient-debugging)
- [Memory Debugging](#memory-debugging)
- [Profiling](#profiling)
- [Numeric Stability](#numeric-stability)

## Common Issues

### NaN/Inf Detection

```python
# Enable anomaly detection (slower, but catches NaN sources)
torch.autograd.set_detect_anomaly(True)

# Check for NaN in tensors
def check_nan(tensor, name="tensor"):
    if torch.isnan(tensor).any():
        raise ValueError(f"NaN detected in {name}")
    if torch.isinf(tensor).any():
        raise ValueError(f"Inf detected in {name}")

# Register hook to check gradients
def check_grad_hook(grad):
    if torch.isnan(grad).any():
        raise ValueError("NaN gradient detected")
    return grad

for param in model.parameters():
    param.register_hook(check_grad_hook)
```

### Shape Mismatch Debugging

```python
# Print shapes at each layer
class ShapeLogger(nn.Module):
    def __init__(self, module: nn.Module, name: str = ""):
        super().__init__()
        self.module = module
        self.name = name

    def forward(self, x):
        print(f"{self.name} input: {x.shape}")
        out = self.module(x)
        print(f"{self.name} output: {out.shape}")
        return out

# Wrap layers for debugging
model.layer1 = ShapeLogger(model.layer1, "layer1")
```

### Device Mismatch

```python
def check_device_consistency(model, inputs):
    model_device = next(model.parameters()).device
    for name, tensor in inputs.items():
        if tensor.device != model_device:
            raise ValueError(
                f"Input '{name}' on {tensor.device}, model on {model_device}"
            )
```

## Gradient Debugging

### Gradient Statistics

```python
def log_gradient_stats(model):
    for name, param in model.named_parameters():
        if param.grad is not None:
            grad = param.grad
            print(f"{name}:")
            print(f"  mean: {grad.mean():.6f}")
            print(f"  std:  {grad.std():.6f}")
            print(f"  max:  {grad.abs().max():.6f}")
            print(f"  norm: {grad.norm():.6f}")
```

### Gradient Flow Visualization

```python
def plot_grad_flow(named_parameters):
    """Plot gradient magnitudes across layers."""
    import matplotlib.pyplot as plt

    ave_grads = []
    layers = []

    for name, param in named_parameters:
        if param.requires_grad and param.grad is not None:
            layers.append(name)
            ave_grads.append(param.grad.abs().mean().item())

    plt.figure(figsize=(12, 4))
    plt.bar(range(len(ave_grads)), ave_grads, alpha=0.7)
    plt.xticks(range(len(ave_grads)), layers, rotation=90)
    plt.xlabel("Layers")
    plt.ylabel("Average Gradient Magnitude")
    plt.title("Gradient Flow")
    plt.tight_layout()
    plt.show()
```

### Dead ReLU Detection

```python
class ReLUMonitor(nn.Module):
    def __init__(self):
        super().__init__()
        self.dead_ratio = 0.0

    def forward(self, x):
        activated = F.relu(x)
        self.dead_ratio = (activated == 0).float().mean().item()
        return activated

# Check after training batch
for name, module in model.named_modules():
    if isinstance(module, ReLUMonitor):
        if module.dead_ratio > 0.9:
            print(f"Warning: {name} has {module.dead_ratio:.1%} dead neurons")
```

## Memory Debugging

### Memory Tracking

```python
def print_memory_usage():
    if torch.cuda.is_available():
        print(f"Allocated: {torch.cuda.memory_allocated() / 1e9:.2f} GB")
        print(f"Cached:    {torch.cuda.memory_reserved() / 1e9:.2f} GB")
        print(f"Max Alloc: {torch.cuda.max_memory_allocated() / 1e9:.2f} GB")

# Reset peak stats
torch.cuda.reset_peak_memory_stats()
```

### Memory Snapshot

```python
# Record memory history for debugging
torch.cuda.memory._record_memory_history()

# ... run your code ...

# Save snapshot
torch.cuda.memory._dump_snapshot("memory_snapshot.pickle")

# Analyze with: python -m torch.cuda.memory_viz memory_snapshot.pickle
```

### Finding Memory Leaks

```python
import gc

def find_tensors():
    """Find all tensors in memory."""
    tensors = []
    for obj in gc.get_objects():
        try:
            if torch.is_tensor(obj):
                tensors.append((type(obj), obj.size(), obj.device))
            elif hasattr(obj, "data") and torch.is_tensor(obj.data):
                tensors.append((type(obj), obj.data.size(), obj.data.device))
        except:
            pass
    return tensors

# Compare before and after
before = find_tensors()
# ... run code ...
after = find_tensors()
print(f"New tensors: {len(after) - len(before)}")
```

## Profiling

### PyTorch Profiler

```python
from torch.profiler import profile, ProfilerActivity, tensorboard_trace_handler

with profile(
    activities=[ProfilerActivity.CPU, ProfilerActivity.CUDA],
    schedule=torch.profiler.schedule(wait=1, warmup=1, active=3, repeat=1),
    on_trace_ready=tensorboard_trace_handler("./log/profiler"),
    record_shapes=True,
    profile_memory=True,
    with_stack=True,
) as prof:
    for step, batch in enumerate(train_loader):
        if step >= 5:
            break
        train_step(batch)
        prof.step()

# Print summary
print(prof.key_averages().table(sort_by="cuda_time_total", row_limit=10))
```

### Timing Utilities

```python
import time

class Timer:
    def __init__(self, name: str = "", sync_cuda: bool = True):
        self.name = name
        self.sync_cuda = sync_cuda

    def __enter__(self):
        if self.sync_cuda and torch.cuda.is_available():
            torch.cuda.synchronize()
        self.start = time.perf_counter()
        return self

    def __exit__(self, *args):
        if self.sync_cuda and torch.cuda.is_available():
            torch.cuda.synchronize()
        self.elapsed = time.perf_counter() - self.start
        if self.name:
            print(f"{self.name}: {self.elapsed*1000:.2f}ms")

# Usage
with Timer("forward pass"):
    output = model(input)
```

### CUDA Events for Precise Timing

```python
start = torch.cuda.Event(enable_timing=True)
end = torch.cuda.Event(enable_timing=True)

start.record()
output = model(input)
end.record()

torch.cuda.synchronize()
print(f"Time: {start.elapsed_time(end):.2f}ms")
```

## Numeric Stability

### Safe Operations

```python
# Safe log
def safe_log(x, eps=1e-8):
    return torch.log(x.clamp(min=eps))

# Safe division
def safe_div(a, b, eps=1e-8):
    return a / (b + eps)

# Stable softmax (already stable in PyTorch, but for reference)
def stable_softmax(x, dim=-1):
    x_max = x.max(dim=dim, keepdim=True).values
    exp_x = torch.exp(x - x_max)
    return exp_x / exp_x.sum(dim=dim, keepdim=True)

# Log-sum-exp (stable)
def log_sum_exp(x, dim=-1):
    return torch.logsumexp(x, dim=dim)  # Built-in is stable
```

### Gradient Clipping Strategies

```python
# Clip by norm (most common)
torch.nn.utils.clip_grad_norm_(model.parameters(), max_norm=1.0)

# Clip by value
torch.nn.utils.clip_grad_value_(model.parameters(), clip_value=1.0)

# Adaptive clipping
def adaptive_clip_grad(parameters, percentile=10):
    grads = [p.grad for p in parameters if p.grad is not None]
    norms = [g.norm().item() for g in grads]
    clip_value = np.percentile(norms, 100 - percentile)
    torch.nn.utils.clip_grad_norm_(parameters, max_norm=clip_value)
```

### Loss Scaling for Mixed Precision

```python
# Manual loss scaling (if not using GradScaler)
scale = 2**16
loss = loss * scale
loss.backward()

for param in model.parameters():
    if param.grad is not None:
        param.grad.data /= scale
```
