# Advanced Training Patterns

## Table of Contents

- [Learning Rate Schedules](#learning-rate-schedules)
- [Loss Functions](#loss-functions)
- [Regularization Techniques](#regularization-techniques)
- [Early Stopping](#early-stopping)
- [Multi-Task Learning](#multi-task-learning)
- [Knowledge Distillation](#knowledge-distillation)

## Learning Rate Schedules

### Warmup + Decay

```python
from torch.optim.lr_scheduler import LambdaLR

def get_scheduler(optimizer, warmup_steps, total_steps):
    def lr_lambda(step):
        if step < warmup_steps:
            return step / warmup_steps
        return max(0.0, (total_steps - step) / (total_steps - warmup_steps))

    return LambdaLR(optimizer, lr_lambda)
```

### Cosine Annealing with Warmup

```python
import math

def get_cosine_schedule_with_warmup(optimizer, warmup_steps, total_steps):
    def lr_lambda(step):
        if step < warmup_steps:
            return step / warmup_steps
        progress = (step - warmup_steps) / (total_steps - warmup_steps)
        return max(0.0, 0.5 * (1.0 + math.cos(math.pi * progress)))

    return LambdaLR(optimizer, lr_lambda)
```

### OneCycleLR

```python
scheduler = torch.optim.lr_scheduler.OneCycleLR(
    optimizer,
    max_lr=1e-3,
    total_steps=len(train_loader) * num_epochs,
    pct_start=0.1,  # Warmup fraction
    anneal_strategy="cos",
)

# Step after each batch, not epoch
for batch in train_loader:
    # ... training
    scheduler.step()
```

## Loss Functions

### Label Smoothing

```python
class LabelSmoothingLoss(nn.Module):
    def __init__(self, num_classes: int, smoothing: float = 0.1):
        super().__init__()
        self.smoothing = smoothing
        self.num_classes = num_classes

    def forward(self, pred: torch.Tensor, target: torch.Tensor) -> torch.Tensor:
        confidence = 1.0 - self.smoothing
        smooth_value = self.smoothing / (self.num_classes - 1)

        one_hot = torch.full_like(pred, smooth_value)
        one_hot.scatter_(1, target.unsqueeze(1), confidence)

        log_probs = F.log_softmax(pred, dim=1)
        return -(one_hot * log_probs).sum(dim=1).mean()

# Or use built-in
criterion = nn.CrossEntropyLoss(label_smoothing=0.1)
```

### Focal Loss

```python
class FocalLoss(nn.Module):
    def __init__(self, alpha: float = 1.0, gamma: float = 2.0):
        super().__init__()
        self.alpha = alpha
        self.gamma = gamma

    def forward(self, pred: torch.Tensor, target: torch.Tensor) -> torch.Tensor:
        ce_loss = F.cross_entropy(pred, target, reduction="none")
        pt = torch.exp(-ce_loss)
        focal_loss = self.alpha * (1 - pt) ** self.gamma * ce_loss
        return focal_loss.mean()
```

### Contrastive Loss (InfoNCE)

```python
def info_nce_loss(features: torch.Tensor, temperature: float = 0.07) -> torch.Tensor:
    """
    Args:
        features: [2N, D] where first N are anchors, second N are positives
    """
    batch_size = features.shape[0] // 2
    labels = torch.arange(batch_size, device=features.device)
    labels = torch.cat([labels + batch_size, labels])  # Positive pairs

    features = F.normalize(features, dim=1)
    similarity = features @ features.T / temperature

    # Mask self-similarity
    mask = torch.eye(2 * batch_size, device=features.device, dtype=torch.bool)
    similarity.masked_fill_(mask, float("-inf"))

    return F.cross_entropy(similarity, labels)
```

## Regularization Techniques

### Dropout Variants

```python
# Standard dropout
nn.Dropout(p=0.1)

# Dropout for 2D feature maps
nn.Dropout2d(p=0.1)

# Alpha dropout (for SELU activations)
nn.AlphaDropout(p=0.1)
```

### Stochastic Depth

```python
class StochasticDepth(nn.Module):
    def __init__(self, drop_prob: float = 0.1):
        super().__init__()
        self.drop_prob = drop_prob

    def forward(self, x: torch.Tensor) -> torch.Tensor:
        if not self.training or self.drop_prob == 0.0:
            return x

        keep_prob = 1.0 - self.drop_prob
        shape = (x.shape[0],) + (1,) * (x.ndim - 1)
        mask = torch.bernoulli(torch.full(shape, keep_prob, device=x.device))
        return x * mask / keep_prob
```

### Mixup and CutMix

```python
def mixup_data(x: torch.Tensor, y: torch.Tensor, alpha: float = 0.2):
    lam = np.random.beta(alpha, alpha) if alpha > 0 else 1.0
    batch_size = x.size(0)
    index = torch.randperm(batch_size, device=x.device)

    mixed_x = lam * x + (1 - lam) * x[index]
    y_a, y_b = y, y[index]
    return mixed_x, y_a, y_b, lam

def mixup_criterion(criterion, pred, y_a, y_b, lam):
    return lam * criterion(pred, y_a) + (1 - lam) * criterion(pred, y_b)
```

## Early Stopping

```python
class EarlyStopping:
    def __init__(self, patience: int = 10, min_delta: float = 0.0):
        self.patience = patience
        self.min_delta = min_delta
        self.counter = 0
        self.best_score = None
        self.should_stop = False

    def __call__(self, val_loss: float) -> bool:
        score = -val_loss

        if self.best_score is None:
            self.best_score = score
        elif score < self.best_score + self.min_delta:
            self.counter += 1
            if self.counter >= self.patience:
                self.should_stop = True
        else:
            self.best_score = score
            self.counter = 0

        return self.should_stop
```

## Multi-Task Learning

### Shared Encoder Pattern

```python
class MultiTaskModel(nn.Module):
    def __init__(self, encoder: nn.Module, task_heads: dict[str, nn.Module]):
        super().__init__()
        self.encoder = encoder
        self.task_heads = nn.ModuleDict(task_heads)

    def forward(self, x: torch.Tensor, task: str) -> torch.Tensor:
        features = self.encoder(x)
        return self.task_heads[task](features)

# Usage
model = MultiTaskModel(
    encoder=Encoder(),
    task_heads={
        "classification": nn.Linear(hidden_dim, num_classes),
        "regression": nn.Linear(hidden_dim, 1),
    }
)
```

### Uncertainty Weighting

```python
class MultiTaskLoss(nn.Module):
    def __init__(self, num_tasks: int):
        super().__init__()
        # Log variance parameters (learnable)
        self.log_vars = nn.Parameter(torch.zeros(num_tasks))

    def forward(self, losses: list[torch.Tensor]) -> torch.Tensor:
        total = 0
        for i, loss in enumerate(losses):
            precision = torch.exp(-self.log_vars[i])
            total += precision * loss + self.log_vars[i]
        return total
```

## Knowledge Distillation

```python
def distillation_loss(
    student_logits: torch.Tensor,
    teacher_logits: torch.Tensor,
    targets: torch.Tensor,
    temperature: float = 4.0,
    alpha: float = 0.5,
) -> torch.Tensor:
    """
    Combines soft targets from teacher with hard targets.
    """
    soft_loss = F.kl_div(
        F.log_softmax(student_logits / temperature, dim=1),
        F.softmax(teacher_logits / temperature, dim=1),
        reduction="batchmean",
    ) * (temperature ** 2)

    hard_loss = F.cross_entropy(student_logits, targets)

    return alpha * soft_loss + (1 - alpha) * hard_loss
```

### Feature Distillation

```python
class FeatureDistillationLoss(nn.Module):
    def __init__(self, student_dim: int, teacher_dim: int):
        super().__init__()
        self.projector = nn.Linear(student_dim, teacher_dim)

    def forward(
        self,
        student_features: torch.Tensor,
        teacher_features: torch.Tensor,
    ) -> torch.Tensor:
        projected = self.projector(student_features)
        return F.mse_loss(projected, teacher_features.detach())
```
