# Security Testing

## Authentication Tests

```typescript
describe('Authentication Security', () => {
  it('rejects invalid credentials', async () => {
    await request(app)
      .post('/api/login')
      .send({ email: 'user@test.com', password: 'wrong' })
      .expect(401);
  });

  it('rejects expired tokens', async () => {
    const expiredToken = createExpiredToken();
    await request(app)
      .get('/api/protected')
      .set('Authorization', `Bearer ${expiredToken}`)
      .expect(401);
  });

  it('rejects tampered tokens', async () => {
    const tamperedToken = validToken.slice(0, -5) + 'xxxxx';
    await request(app)
      .get('/api/protected')
      .set('Authorization', `Bearer ${tamperedToken}`)
      .expect(401);
  });

  it('enforces rate limiting on login', async () => {
    for (let i = 0; i < 6; i++) {
      await request(app)
        .post('/api/login')
        .send({ email: 'user@test.com', password: 'wrong' });
    }

    await request(app)
      .post('/api/login')
      .send({ email: 'user@test.com', password: 'correct' })
      .expect(429);
  });
});
```

## Authorization Tests

```typescript
describe('Authorization', () => {
  it('denies access to other users resources', async () => {
    await request(app)
      .get('/api/users/other-user-id/data')
      .set('Authorization', `Bearer ${userAToken}`)
      .expect(403);
  });

  it('denies admin routes to regular users', async () => {
    await request(app)
      .delete('/api/admin/users/123')
      .set('Authorization', `Bearer ${regularUserToken}`)
      .expect(403);
  });
});
```

## Input Validation Tests

```typescript
describe('Input Validation', () => {
  it('rejects SQL injection attempts', async () => {
    await request(app)
      .get('/api/users')
      .query({ search: "'; DROP TABLE users; --" })
      .expect(400);
  });

  it('rejects XSS in input fields', async () => {
    const response = await request(app)
      .post('/api/posts')
      .send({ title: '<script>alert("xss")</script>' })
      .expect(201);

    expect(response.body.title).not.toContain('<script>');
  });

  it('validates file upload types', async () => {
    await request(app)
      .post('/api/upload')
      .attach('file', 'malicious.exe')
      .expect(400);
  });
});
```

## Security Headers Test

```typescript
describe('Security Headers', () => {
  it('sets security headers', async () => {
    const response = await request(app).get('/');

    expect(response.headers['x-content-type-options']).toBe('nosniff');
    expect(response.headers['x-frame-options']).toBe('DENY');
    expect(response.headers['strict-transport-security']).toBeDefined();
  });
});
```

## Security Test Checklist

| Category | Tests |
|----------|-------|
| **Auth** | Invalid creds, token expiry, tampering |
| **Input** | SQL injection, XSS, command injection |
| **Access** | IDOR, privilege escalation |
| **Rate Limit** | Brute force, API abuse |
| **Headers** | CSP, HSTS, X-Frame-Options |
| **Data** | PII exposure, error messages |

## Quick Reference

| Vulnerability | Test Approach |
|---------------|---------------|
| SQL Injection | `'; DROP TABLE--` in inputs |
| XSS | `<script>alert(1)</script>` |
| IDOR | Access other user's resources |
| CSRF | Missing/invalid tokens |
| Auth Bypass | Missing auth, expired tokens |
