"""
Python Worker Entry Point

This is the main entry point for your Python Worker.
Uses the WorkerEntrypoint class pattern (required as of Aug 2025).
"""

from workers import WorkerEntrypoint, Response
import json


class Default(WorkerEntrypoint):
    """
    Main Worker class.

    Handlers:
    - fetch: Handle HTTP requests
    - Access bindings via self.env (e.g., self.env.DB, self.env.MY_KV)
    """

    async def fetch(self, request):
        """
        Handle incoming HTTP requests.

        Args:
            request: The incoming Request object

        Returns:
            Response object
        """
        # Get request details
        url = request.url
        method = request.method

        # Route based on path
        path = url.split("?")[0].rstrip("/")

        if path.endswith("/api/hello"):
            return await self.handle_hello(request)

        if path.endswith("/api/data") and method == "POST":
            return await self.handle_post_data(request)

        # Default response
        return Response(
            json.dumps({
                "message": "Hello from Python Worker!",
                "method": method,
                "url": url,
            }),
            headers={"Content-Type": "application/json"}
        )

    async def handle_hello(self, request):
        """Handle GET /api/hello"""
        # Example: Access query parameters
        # url = URL(request.url)
        # name = url.searchParams.get("name", "World")

        return Response(
            json.dumps({"greeting": "Hello, World!"}),
            headers={"Content-Type": "application/json"}
        )

    async def handle_post_data(self, request):
        """Handle POST /api/data"""
        try:
            # Parse JSON body
            body = await request.json()

            # Process the data
            result = {
                "received": body,
                "processed": True
            }

            return Response(
                json.dumps(result),
                headers={"Content-Type": "application/json"}
            )

        except Exception as e:
            return Response(
                json.dumps({"error": str(e)}),
                status=400,
                headers={"Content-Type": "application/json"}
            )


# ============================================
# BINDING EXAMPLES (uncomment as needed)
# ============================================

# async def example_d1_query(self):
#     """Query D1 database"""
#     result = await self.env.DB.prepare(
#         "SELECT * FROM users WHERE id = ?"
#     ).bind(1).first()
#     return result

# async def example_kv_operations(self):
#     """KV storage operations"""
#     # Get
#     value = await self.env.MY_KV.get("key")
#
#     # Put
#     await self.env.MY_KV.put("key", "value")
#
#     # Delete
#     await self.env.MY_KV.delete("key")

# async def example_r2_operations(self):
#     """R2 object storage"""
#     # Get object
#     obj = await self.env.MY_BUCKET.get("file.txt")
#     if obj:
#         content = await obj.text()
#
#     # Put object
#     await self.env.MY_BUCKET.put("file.txt", "content")

# async def example_workers_ai(self):
#     """Workers AI inference"""
#     response = await self.env.AI.run(
#         "@cf/meta/llama-2-7b-chat-int8",
#         {"prompt": "Hello!"}
#     )
#     return response
