/**
 * Google Chat Webhook Handler (Cloudflare Worker)
 *
 * Simplest integration - receives notifications from external systems
 * and posts them to Google Chat spaces via incoming webhooks.
 *
 * No authentication required for incoming webhooks.
 */

interface Env {
  // Webhook URLs stored as secrets
  CI_CD_WEBHOOK_URL: string
  ALERTS_WEBHOOK_URL: string
}

export default {
  async fetch(request: Request, env: Env): Promise<Response> {
    if (request.method !== 'POST') {
      return new Response('Method not allowed', { status: 405 })
    }

    const { type, message, webhookKey } = await request.json()

    // Select webhook URL based on type
    const webhookUrl = webhookKey === 'cicd'
      ? env.CI_CD_WEBHOOK_URL
      : env.ALERTS_WEBHOOK_URL

    // Send to Google Chat
    const response = await fetch(webhookUrl, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        text: message,
        cardsV2: [{
          cardId: `notif-${Date.now()}`,
          card: {
            header: {
              title: type === 'success' ? '✅ Success' : '⚠️ Alert',
              subtitle: new Date().toLocaleString()
            },
            sections: [{
              widgets: [{
                textParagraph: { text: message }
              }]
            }]
          }
        }]
      })
    })

    return new Response(
      JSON.stringify({ success: response.ok }),
      { headers: { 'Content-Type': 'application/json' } }
    )
  }
}
