# New Architecture Common Errors

**Last Updated:** 2025-11-22
**React Native Versions:** 0.76+ (New Architecture introduced as default)
**Source:** [New Architecture Migration Guide](https://reactnative.dev/docs/new-architecture-intro)

---

## Error Categories

1. [Build Errors](#build-errors)
2. [Runtime Errors](#runtime-errors)
3. [Library Compatibility](#library-compatibility)
4. [iOS Specific](#ios-specific)
5. [Android Specific](#android-specific)

---

## Build Errors

### 1. "C++11 too old" / "targeting C++11"

**Full Error:**
```
error: 'if constexpr' is a C++17 extension
error: targeting C++11 but using C++17 features
```

**Why It Happens:**
New Architecture requires C++17 or newer. Old projects default to C++11.

**Fix (Android):**
```gradle
// android/app/build.gradle
android {
    ...
    compileOptions {
        sourceCompatibility JavaVersion.VERSION_17
        targetCompatibility JavaVersion.VERSION_17
    }

    kotlinOptions {
        jvmTarget = '17'
    }
}
```

**Fix (iOS):**
```ruby
# ios/Podfile
post_install do |installer|
  installer.pods_project.targets.each do |target|
    target.build_configurations.each do |config|
      config.build_settings['CLANG_CXX_LANGUAGE_STANDARD'] = 'c++20'
    end
  end
end
```

---

### 2. "glog module import error"

**Full Error:**
```
error: import of module 'glog.glog.log_severity' appears within namespace 'google'
```

**Why It Happens:**
Conflict between glog module and React Native's internal usage.

**Fix (iOS):**
```ruby
# ios/Podfile
# Add this before other config
use_frameworks! :linkage => :static

# Then clean and reinstall
cd ios
rm -rf Pods Podfile.lock
pod install
```

**Alternative Fix:**
```ruby
# ios/Podfile
post_install do |installer|
  installer.pods_project.targets.each do |target|
    if target.name == 'glog'
      target.build_configurations.each do |config|
        config.build_settings['CLANG_ALLOW_NON_MODULAR_INCLUDES_IN_FRAMEWORK_MODULES'] = 'YES'
      end
    end
  end
end
```

---

### 3. "AppDelegate migration required"

**Full Error:**
```
error: RCTAppDependencyProvider not found
Undefined symbol: RCTAppDependencyProvider
```

**Why It Happens:**
Upgrading from Objective-C to Swift iOS template without adding required provider.

**Fix:**
Add this line to `AppDelegate.swift`:

```swift
import React
import ReactCoreModules

@main
class AppDelegate: UIResponder, UIApplicationDelegate {
    func application(_ application: UIApplication,
                     didFinishLaunchingWithOptions launchOptions: [UIApplication.LaunchOptionsKey: Any]?) -> Bool {

        // ⚠️ CRITICAL: Add this line
        RCTAppDependencyProvider.sharedInstance()

        return true
    }
}
```

---

## Runtime Errors

### 4. "Fabric component descriptor not found"

**Full Error:**
```
Fabric component descriptor provider not found for component: MyComponent
```

**Why It Happens:**
Component/library not compatible with Fabric (New Architecture's rendering system).

**Fix Options:**

**A) Update library** to New Architecture version:
```bash
# Check library docs for New Architecture support
npm update <library-name>@latest
```

**B) Use interop layer** (0.76-0.81 only):
```bash
# Android (gradle.properties)
newArchEnabled=true
interopEnabled=true

# iOS - interop automatic
```

**C) Temporarily disable New Architecture** (0.76-0.81 only):
```bash
# gradle.properties
newArchEnabled=false

# iOS
RCT_NEW_ARCH_ENABLED=0 pod install
```

⚠️ **Note:** Cannot disable in 0.82+ - New Architecture is mandatory.

---

### 5. "TurboModule not registered"

**Full Error:**
```
TurboModule 'ModuleName' not found
TurboModule registry not available
```

**Why It Happens:**
Native module not compatible with TurboModules (New Architecture's native module system).

**Fix:**

**A) Update library**:
```bash
npm update <library-name>@latest
```

**B) Check library compatibility**:
```bash
# Look for "New Architecture" or "Fabric" in docs
# Check GitHub issues for compatibility status
```

**C) Use bridge interop** (0.76-0.81):
Most libraries work via bridge interop during transition period.

**D) Downgrade if critical** (last resort):
```bash
# Find last pre-New-Arch version
npm install <library-name>@<old-version>
```

---

### 6. "Bridge not available"

**Full Error:**
```
RCTBridge required for this functionality
Cannot access bridge module
```

**Why It Happens:**
Code trying to use legacy bridge APIs in New Architecture.

**Fix:**

Replace bridge-dependent code:

```typescript
// ❌ OLD (uses bridge)
import { NativeModules } from 'react-native';
const { MyModule } = NativeModules;

// ✅ NEW (uses TurboModules)
import { TurboModuleRegistry } from 'react-native';
const MyModule = TurboModuleRegistry.get('MyModule');
```

---

## Library Compatibility

### 7. Redux crashes on store creation

**Error:**
```
TypeError: Cannot read property 'dispatch' of undefined
Redux store crashes during initialization
```

**Why It Happens:**
Old `redux` + `redux-thunk` packages incompatible with New Architecture.

**Fix:**
```bash
# Remove old Redux
npm uninstall redux redux-thunk

# Install Redux Toolkit (compatible)
npm install @reduxjs/toolkit react-redux
```

**Migration:**
```typescript
// ❌ OLD
import { createStore, applyMiddleware } from 'redux';
import thunk from 'redux-thunk';
const store = createStore(reducer, applyMiddleware(thunk));

// ✅ NEW
import { configureStore } from '@reduxjs/toolkit';
const store = configureStore({
  reducer: rootReducer,
  // Thunk included by default
});
```

---

### 8. i18n-js unreliable

**Error:**
```
Translations not updating
App crashes when changing locale
```

**Why It Happens:**
`i18n-js` not fully compatible with New Architecture.

**Fix:**
```bash
# Remove i18n-js
npm uninstall i18n-js

# Install react-i18next (compatible)
npm install react-i18next i18next
```

**Setup:**
```typescript
import i18n from 'i18next';
import { initReactI18next } from 'react-i18next';

i18n
  .use(initReactI18next)
  .init({
    resources: {
      en: { translation: { ... } },
      es: { translation: { ... } }
    },
    lng: 'en',
    fallbackLng: 'en'
  });

// Usage
import { useTranslation } from 'react-i18next';

function MyComponent() {
  const { t } = useTranslation();
  return <Text>{t('welcome')}</Text>;
}
```

---

### 9. CodePush crashes on Android

**Error:**
```
Android app crashes looking for bundle named 'null'
CodePush update fails silently
```

**Why It Happens:**
Known incompatibility between CodePush and New Architecture.

**Fix:**

**A) Disable CodePush** (recommended until fixed):
```bash
npm uninstall react-native-code-push
```

**B) Monitor GitHub** for official support:
- https://github.com/microsoft/react-native-code-push/issues

**C) Use alternatives**:
- Expo Updates (if using Expo)
- Native OTA update solutions

---

## iOS Specific

### 10. Hermes symbol conflicts

**Error:**
```
duplicate symbol '_OBJC_CLASS_$_HermesExecutorFactory'
```

**Why It Happens:**
Multiple Hermes versions or incorrect Podfile configuration.

**Fix:**
```ruby
# ios/Podfile
use_react_native!(
  :path => config[:reactNativePath],
  :hermes_enabled => true,  # Ensure enabled
  :fabric_enabled => true
)

# Then clean
cd ios
rm -rf Pods Podfile.lock
pod install --repo-update
```

---

### 11. "NSUnknownKeyException"

**Error:**
```
NSUnknownKeyException: this class is not key value coding-compliant for the key 'reactViewTag'
```

**Why It Happens:**
Old view manager code trying to access deprecated properties.

**Fix:**

Update native view manager to use Fabric APIs:

```objective-c
// ❌ OLD
RCTSetViewManager.m uses reactViewTag

// ✅ NEW
// Use Fabric's componentDescriptorProvider instead
// See: https://reactnative.dev/docs/new-architecture-library-ios
```

---

## Android Specific

### 12. Gradle build fails with "Cannot resolve symbol"

**Error:**
```
error: cannot find symbol: ReactInstanceManager
error: package com.facebook.react.bridge does not exist
```

**Why It Happens:**
Missing New Architecture dependencies in `build.gradle`.

**Fix:**
```gradle
// android/app/build.gradle
dependencies {
    implementation("com.facebook.react:react-android")
    implementation("com.facebook.react:react-native-codegen")
    implementation("com.facebook.react:hermes-android")

    if (hermesEnabled.toBoolean()) {
        implementation("com.facebook.react:hermes-android")
    } else {
        implementation jscFlavor
    }
}
```

---

### 13. "Failed to load native library"

**Error:**
```
java.lang.UnsatisfiedLinkError: couldn't find libreactnativejni.so
```

**Why It Happens:**
Native libraries not built correctly for New Architecture.

**Fix:**
```bash
# Clean and rebuild
cd android
./gradlew clean
cd ..
npm run android

# If still failing, clear all caches:
cd android
rm -rf .gradle build app/build
cd ..
rm -rf node_modules
npm install
npm run android
```

---

## Quick Diagnostic Checklist

When you encounter a New Architecture error:

- [ ] **Check React Native version** - Is it 0.76+?
- [ ] **Check if New Architecture is enabled**:
  - Android: `grep newArchEnabled android/gradle.properties`
  - iOS: `grep RCT_NEW_ARCH_ENABLED ios/Podfile`
- [ ] **Check React version** - Should be 19+ for RN 0.78+
- [ ] **Check C++ version** - Should be C++17 or C++20
- [ ] **Check library compatibility** - Search "[library] new architecture"
- [ ] **Try interop layer** (0.76-0.81 only) - May help with incompatible libraries
- [ ] **Check GitHub issues** - Library may have known issues
- [ ] **Try clean rebuild**:
  ```bash
  # iOS
  cd ios && rm -rf Pods Podfile.lock && pod install

  # Android
  cd android && ./gradlew clean

  # Both
  rm -rf node_modules && npm install
  ```

---

## Resources

- **New Architecture Intro**: https://reactnative.dev/docs/new-architecture-intro
- **iOS Migration**: https://reactnative.dev/docs/new-architecture-library-ios
- **Android Migration**: https://reactnative.dev/docs/new-architecture-library-android
- **Compatibility Tracker**: https://github.com/reactwg/react-native-new-architecture
- **Upgrade Helper**: https://react-native-community.github.io/upgrade-helper/

---

**Bottom Line:** Most New Architecture errors come from incompatible libraries or missing C++ configuration. Check library compatibility first, update C++ settings, and use the interop layer (0.76-0.81) during migration.
