/**
 * ts-agent-sdk Base HTTP Client
 *
 * Provides common HTTP functionality for all API clients.
 */

import { NetworkError, RateLimitError } from '../errors';

export interface RequestOptions {
  method?: 'GET' | 'POST' | 'PUT' | 'DELETE' | 'PATCH';
  headers?: Record<string, string>;
  body?: unknown;
  timeout?: number;
}

export interface APIResponse<T = unknown> {
  data: T;
  status: number;
  headers: Headers;
}

/**
 * Make an HTTP request with standard error handling.
 */
export async function request<T>(url: string, options: RequestOptions = {}): Promise<APIResponse<T>> {
  const { method = 'GET', headers = {}, body, timeout = 30000 } = options;

  const controller = new AbortController();
  const timeoutId = setTimeout(() => controller.abort(), timeout);

  try {
    const response = await fetch(url, {
      method,
      headers: {
        'Content-Type': 'application/json',
        ...headers,
      },
      body: body ? JSON.stringify(body) : undefined,
      signal: controller.signal,
    });

    clearTimeout(timeoutId);

    // Handle rate limiting
    if (response.status === 429) {
      const retryAfter = parseInt(response.headers.get('Retry-After') || '60', 10);
      throw new RateLimitError(retryAfter);
    }

    // Parse response
    const contentType = response.headers.get('Content-Type') || '';
    let data: T;

    if (contentType.includes('application/json')) {
      data = await response.json();
    } else {
      data = (await response.text()) as T;
    }

    // Check for error status
    if (!response.ok) {
      const errorMessage = typeof data === 'object' && data !== null && 'error' in data
        ? (data as { error: { message?: string } }).error?.message || JSON.stringify(data)
        : String(data);
      throw new Error(`API error (${response.status}): ${errorMessage}`);
    }

    return {
      data,
      status: response.status,
      headers: response.headers,
    };
  } catch (error) {
    clearTimeout(timeoutId);

    if (error instanceof RateLimitError) {
      throw error;
    }

    if (error instanceof Error) {
      if (error.name === 'AbortError') {
        throw new NetworkError(`Request timed out after ${timeout}ms`);
      }
      if (error.message.includes('API error')) {
        throw error;
      }
      throw new NetworkError(error.message, error);
    }

    throw new NetworkError('Unknown error occurred');
  }
}

/**
 * Make a GET request.
 */
export async function get<T>(url: string, headers?: Record<string, string>): Promise<T> {
  const response = await request<T>(url, { method: 'GET', headers });
  return response.data;
}

/**
 * Make a POST request.
 */
export async function post<T>(url: string, body: unknown, headers?: Record<string, string>): Promise<T> {
  const response = await request<T>(url, { method: 'POST', body, headers });
  return response.data;
}
