/**
 * ts-agent-sdk API Configuration
 *
 * Environment variables for external API access:
 * - SDK_GEMINI_API_KEY: Google Gemini API key
 * - SDK_WORKERS_AI_ACCOUNT_ID: Cloudflare account ID for Workers AI
 * - SDK_WORKERS_AI_API_TOKEN: Cloudflare API token for Workers AI
 * - SDK_SLACK_WEBHOOK_URL: Slack incoming webhook URL
 */

export interface APIConfig {
  gemini: {
    apiKey: string;
    defaultModel: string;
  };
  workersAI: {
    accountId: string;
    apiToken: string;
  };
  slack: {
    webhookUrl: string;
  };
}

/**
 * Load API configuration from environment variables.
 */
export function loadAPIConfig(): APIConfig {
  return {
    gemini: {
      apiKey: process.env.SDK_GEMINI_API_KEY || '',
      defaultModel: process.env.SDK_GEMINI_MODEL || 'gemini-2.5-flash',
    },
    workersAI: {
      accountId: process.env.SDK_WORKERS_AI_ACCOUNT_ID || process.env.SDK_CF_ACCOUNT_ID || '',
      apiToken: process.env.SDK_WORKERS_AI_API_TOKEN || process.env.SDK_CF_API_TOKEN || '',
    },
    slack: {
      webhookUrl: process.env.SDK_SLACK_WEBHOOK_URL || '',
    },
  };
}

/**
 * Validate that required API keys are present.
 */
export function validateGeminiConfig(config: APIConfig): void {
  if (!config.gemini.apiKey) {
    throw new Error('SDK_GEMINI_API_KEY is required for Gemini API calls.');
  }
}

export function validateWorkersAIConfig(config: APIConfig): void {
  if (!config.workersAI.accountId) {
    throw new Error('SDK_WORKERS_AI_ACCOUNT_ID is required for Workers AI calls.');
  }
  if (!config.workersAI.apiToken) {
    throw new Error('SDK_WORKERS_AI_API_TOKEN is required for Workers AI calls.');
  }
}

export function validateSlackConfig(config: APIConfig): void {
  if (!config.slack.webhookUrl) {
    throw new Error('SDK_SLACK_WEBHOOK_URL is required for Slack notifications.');
  }
}
