/**
 * ts-agent-sdk Slack Client
 *
 * Send notifications via Slack incoming webhooks.
 *
 * Environment Variables:
 * - SDK_SLACK_WEBHOOK_URL: Slack incoming webhook URL
 */

import { post } from './base';
import { loadAPIConfig, validateSlackConfig } from './config';

export interface SlackBlock {
  type: 'section' | 'divider' | 'header' | 'context' | 'actions';
  text?: {
    type: 'mrkdwn' | 'plain_text';
    text: string;
    emoji?: boolean;
  };
  fields?: {
    type: 'mrkdwn' | 'plain_text';
    text: string;
  }[];
  elements?: unknown[];
  accessory?: unknown;
}

export interface SlackMessage {
  text?: string;
  blocks?: SlackBlock[];
  attachments?: unknown[];
  mrkdwn?: boolean;
}

/**
 * Slack Client for sending webhook notifications.
 */
export class SlackClient {
  private webhookUrl: string;

  constructor(webhookUrl?: string) {
    const config = loadAPIConfig();
    this.webhookUrl = webhookUrl || config.slack.webhookUrl;
  }

  /**
   * Post a simple text message.
   */
  async postMessage(text: string): Promise<void> {
    this.ensureConfigured();
    await post<string>(this.webhookUrl, { text });
  }

  /**
   * Post a message with Block Kit blocks for rich formatting.
   * @see https://api.slack.com/block-kit
   */
  async postBlocks(blocks: SlackBlock[], fallbackText?: string): Promise<void> {
    this.ensureConfigured();
    await post<string>(this.webhookUrl, {
      text: fallbackText || 'New message',
      blocks,
    });
  }

  /**
   * Post a formatted notification with title, message, and optional fields.
   */
  async postNotification(options: {
    title: string;
    message: string;
    fields?: { label: string; value: string }[];
    color?: 'good' | 'warning' | 'danger' | string;
  }): Promise<void> {
    const blocks: SlackBlock[] = [
      {
        type: 'header',
        text: { type: 'plain_text', text: options.title, emoji: true },
      },
      {
        type: 'section',
        text: { type: 'mrkdwn', text: options.message },
      },
    ];

    if (options.fields && options.fields.length > 0) {
      blocks.push({
        type: 'section',
        fields: options.fields.map((f) => ({
          type: 'mrkdwn' as const,
          text: `*${f.label}:*\n${f.value}`,
        })),
      });
    }

    await this.postBlocks(blocks, options.title);
  }

  /**
   * Post an error notification.
   */
  async postError(title: string, error: Error | string, context?: Record<string, string>): Promise<void> {
    const errorMessage = error instanceof Error ? error.message : error;
    const stack = error instanceof Error ? error.stack : undefined;

    const blocks: SlackBlock[] = [
      {
        type: 'header',
        text: { type: 'plain_text', text: `🚨 ${title}`, emoji: true },
      },
      {
        type: 'section',
        text: { type: 'mrkdwn', text: `\`\`\`${errorMessage}\`\`\`` },
      },
    ];

    if (context) {
      blocks.push({
        type: 'context',
        elements: Object.entries(context).map(([key, value]) => ({
          type: 'mrkdwn',
          text: `*${key}:* ${value}`,
        })),
      });
    }

    if (stack) {
      blocks.push({
        type: 'section',
        text: { type: 'mrkdwn', text: `<details>\n\`\`\`${stack.slice(0, 500)}\`\`\`\n</details>` },
      });
    }

    await this.postBlocks(blocks, title);
  }

  private ensureConfigured(): void {
    if (!this.webhookUrl) {
      const config = loadAPIConfig();
      validateSlackConfig(config);
      this.webhookUrl = config.slack.webhookUrl;
    }
  }
}

// Default singleton instance
export const slack = new SlackClient();
