/**
 * ts-agent-sdk Workers AI Client
 *
 * Typed wrapper for Cloudflare Workers AI REST API.
 *
 * Environment Variables:
 * - SDK_WORKERS_AI_ACCOUNT_ID or SDK_CF_ACCOUNT_ID: Cloudflare account ID
 * - SDK_WORKERS_AI_API_TOKEN or SDK_CF_API_TOKEN: Cloudflare API token
 */

import { post } from './base';
import { loadAPIConfig, validateWorkersAIConfig } from './config';

/**
 * Workers AI model categories and common models.
 */
export const WORKERS_AI_MODELS = {
  textGeneration: [
    '@cf/meta/llama-3.1-8b-instruct',
    '@cf/meta/llama-3.2-3b-instruct',
    '@cf/mistral/mistral-7b-instruct-v0.2',
    '@hf/google/gemma-7b-it',
  ],
  textEmbeddings: [
    '@cf/baai/bge-base-en-v1.5',
    '@cf/baai/bge-large-en-v1.5',
    '@cf/baai/bge-small-en-v1.5',
  ],
  textClassification: [
    '@cf/huggingface/distilbert-sst-2-int8',
  ],
  imageClassification: [
    '@cf/microsoft/resnet-50',
  ],
  speechToText: [
    '@cf/openai/whisper',
  ],
} as const;

export interface TextGenerationInput {
  prompt: string;
  max_tokens?: number;
  temperature?: number;
  top_p?: number;
  stream?: boolean;
}

export interface TextGenerationResponse {
  response: string;
}

export interface EmbeddingsInput {
  text: string | string[];
}

export interface EmbeddingsResponse {
  data: number[][];
  shape: number[];
}

/**
 * Workers AI Client for running AI models on Cloudflare's edge.
 */
export class WorkersAIClient {
  private accountId: string;
  private apiToken: string;

  constructor() {
    const config = loadAPIConfig();
    this.accountId = config.workersAI.accountId;
    this.apiToken = config.workersAI.apiToken;
  }

  /**
   * Run any Workers AI model with arbitrary input.
   */
  async run<T>(model: string, input: unknown): Promise<T> {
    this.ensureConfigured();

    const url = `https://api.cloudflare.com/client/v4/accounts/${this.accountId}/ai/run/${model}`;

    const response = await post<{ success: boolean; result: T; errors: unknown[] }>(
      url,
      input,
      { Authorization: `Bearer ${this.apiToken}` }
    );

    if (!response.success) {
      throw new Error(`Workers AI error: ${JSON.stringify(response.errors)}`);
    }

    return response.result;
  }

  /**
   * Generate text using a language model.
   */
  async textGeneration(
    prompt: string,
    model: string = WORKERS_AI_MODELS.textGeneration[0],
    options: Omit<TextGenerationInput, 'prompt'> = {}
  ): Promise<string> {
    const result = await this.run<TextGenerationResponse>(model, {
      prompt,
      ...options,
    });
    return result.response;
  }

  /**
   * Generate text embeddings for semantic search.
   */
  async textEmbeddings(
    text: string | string[],
    model: string = WORKERS_AI_MODELS.textEmbeddings[0]
  ): Promise<number[][]> {
    const result = await this.run<EmbeddingsResponse>(model, { text });
    return result.data;
  }

  /**
   * Classify text sentiment or category.
   */
  async textClassification(
    text: string,
    model: string = WORKERS_AI_MODELS.textClassification[0]
  ): Promise<{ label: string; score: number }[]> {
    return this.run<{ label: string; score: number }[]>(model, { text });
  }

  /**
   * List available models.
   */
  async listModels(): Promise<{ id: string; name: string; task: string }[]> {
    this.ensureConfigured();

    const url = `https://api.cloudflare.com/client/v4/accounts/${this.accountId}/ai/models/search`;

    const response = await post<{
      success: boolean;
      result: { id: string; name: string; task: { name: string } }[];
    }>(
      url,
      {},
      { Authorization: `Bearer ${this.apiToken}` }
    );

    return response.result.map((m) => ({
      id: m.id,
      name: m.name,
      task: m.task.name,
    }));
  }

  private ensureConfigured(): void {
    const config = loadAPIConfig();
    validateWorkersAIConfig(config);
    this.accountId = config.workersAI.accountId;
    this.apiToken = config.workersAI.apiToken;
  }
}

// Default singleton instance
export const workersAI = new WorkersAIClient();
