/**
 * ts-agent-sdk MCP Client
 *
 * Wraps MCP JSON-RPC calls with typed interfaces and error handling.
 */

import { loadConfig, validateConfig, type SDKConfig } from './config';
import {
  SDKError,
  AuthError,
  RateLimitError,
  MCPError,
  NetworkError,
} from './errors';

/**
 * Standard response wrapper from MCP tools.
 * Tools return { success: true, ...data } or { success: false, error: string }
 */
export interface MCPResponse<T = unknown> {
  success: boolean;
  data?: T;
  error?: string;
}

/**
 * JSON-RPC 2.0 response structure
 */
interface JsonRpcResponse {
  jsonrpc: '2.0';
  id: number;
  result?: {
    content?: Array<{ type: string; text: string }>;
    isError?: boolean;
  };
  error?: {
    code: number;
    message: string;
  };
}

/**
 * MCP Client for making typed tool calls.
 *
 * Usage:
 * ```typescript
 * const client = new MCPClient();
 * const result = await client.callTool('/api/mcp-docs/message', 'list_spaces', {});
 * ```
 */
export class MCPClient {
  private config: SDKConfig;
  private requestId = 0;

  constructor(config?: Partial<SDKConfig>) {
    this.config = { ...loadConfig(), ...config };
    validateConfig(this.config);
  }

  /**
   * Call an MCP tool with typed input and output.
   *
   * @param endpoint - The MCP endpoint path (e.g., '/api/mcp-docs/message')
   * @param toolName - The tool name (e.g., 'create_document')
   * @param args - Tool arguments matching the tool's input schema
   * @returns The tool's response data
   * @throws {AuthError} If authentication fails
   * @throws {RateLimitError} If rate limited
   * @throws {MCPError} If the tool returns an error
   * @throws {NetworkError} If the request fails
   */
  async callTool<TInput extends Record<string, unknown>, TOutput>(
    endpoint: string,
    toolName: string,
    args: TInput
  ): Promise<TOutput> {
    const url = `${this.config.baseUrl}${endpoint}`;

    const payload = {
      jsonrpc: '2.0' as const,
      method: 'tools/call',
      params: {
        name: toolName,
        arguments: args,
      },
      id: ++this.requestId,
    };

    let response: Response;

    try {
      response = await fetch(url, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          ...(this.config.apiToken && {
            Authorization: `Bearer ${this.config.apiToken}`,
          }),
        },
        body: JSON.stringify(payload),
      });
    } catch (error) {
      throw new NetworkError(
        `Failed to connect to ${url}`,
        error instanceof Error ? error : undefined
      );
    }

    // Handle HTTP-level errors
    if (response.status === 401) {
      throw new AuthError();
    }

    if (response.status === 429) {
      const retryAfter = parseInt(response.headers.get('Retry-After') || '60', 10);
      throw new RateLimitError(retryAfter);
    }

    if (!response.ok) {
      const text = await response.text().catch(() => 'Unknown error');
      throw new SDKError(`HTTP ${response.status}: ${text}`, `HTTP_${response.status}`);
    }

    // Parse JSON-RPC response
    const json: JsonRpcResponse = await response.json();

    // Handle JSON-RPC level errors
    if (json.error) {
      throw new MCPError(json.error.message, json.error.code);
    }

    // Extract tool result from MCP response format
    const result = json.result;
    if (!result?.content?.[0]) {
      throw new MCPError('Empty response from MCP server');
    }

    const content = result.content[0];
    if (content.type !== 'text') {
      throw new MCPError(`Unexpected content type: ${content.type}`);
    }

    // Parse the JSON string inside the text content
    const parsed = JSON.parse(content.text);

    // Handle tool-level errors
    if (result.isError || parsed.success === false) {
      throw new MCPError(parsed.error || 'Tool execution failed');
    }

    return parsed as TOutput;
  }

  /**
   * Get the current configuration (for debugging).
   */
  getConfig(): Readonly<SDKConfig> {
    return { ...this.config };
  }
}

/**
 * Default client instance.
 * Uses environment variables for configuration.
 */
export const defaultClient = new MCPClient();
