/**
 * ts-agent-sdk Database Client
 *
 * Executes SQL queries against Cloudflare D1.
 * Supports both local (wrangler CLI) and remote (REST API) modes.
 */

import { execSync } from 'child_process';
import { loadDBConfig, resolveDBMode, validateRemoteConfig, type DBConfig } from './config';
import type { D1QueryResult, D1APIResponse, CountOptions, SelectOptions, SQLParam } from './types';

/**
 * Database Client for D1 queries.
 */
export class DBClient {
  private config: DBConfig;
  private effectiveMode: 'local' | 'remote';

  constructor() {
    this.config = loadDBConfig();
    this.effectiveMode = resolveDBMode(this.config);
  }

  /**
   * Get the current execution mode.
   */
  getMode(): 'local' | 'remote' {
    return this.effectiveMode;
  }

  /**
   * Execute a raw SQL query.
   *
   * @param sql - SQL query string
   * @param params - Parameters to bind (for parameterized queries)
   * @returns Array of result rows
   *
   * @example
   * const docs = await db.query<Document>('SELECT * FROM documents WHERE status = ?', ['published']);
   */
  async query<T = Record<string, unknown>>(sql: string, params?: SQLParam[]): Promise<T[]> {
    if (this.effectiveMode === 'local') {
      return this.executeLocal<T>(sql, params);
    } else {
      return this.executeRemote<T>(sql, params);
    }
  }

  /**
   * Execute a SQL statement that modifies data.
   *
   * @param sql - SQL statement (INSERT, UPDATE, DELETE)
   * @param params - Parameters to bind
   * @returns Number of rows affected
   */
  async execute(sql: string, params?: SQLParam[]): Promise<{ changes: number; lastRowId?: number }> {
    const result = await this.queryRaw<Record<string, unknown>>(sql, params);
    return {
      changes: result.meta?.changes || 0,
      lastRowId: result.meta?.last_row_id,
    };
  }

  /**
   * Execute query and return raw D1 result with metadata.
   */
  async queryRaw<T = Record<string, unknown>>(sql: string, params?: SQLParam[]): Promise<D1QueryResult<T>> {
    if (this.effectiveMode === 'local') {
      return this.executeLocalRaw<T>(sql, params);
    } else {
      return this.executeRemoteRaw<T>(sql, params);
    }
  }

  /**
   * Count rows in a table.
   *
   * @param table - Table name
   * @param options - Optional where clause
   *
   * @example
   * const totalDocs = await db.count('documents');
   * const publishedDocs = await db.count('documents', { where: { status: 'published' } });
   */
  async count(table: string, options: CountOptions = {}): Promise<number> {
    const { whereClause, params } = this.buildWhereClause(options.where);
    const sql = `SELECT COUNT(*) as count FROM ${this.escapeIdentifier(table)}${whereClause}`;
    const result = await this.query<{ count: number }>(sql, params);
    return result[0]?.count || 0;
  }

  /**
   * Select rows from a table.
   *
   * @param table - Table name
   * @param options - Select options (columns, where, orderBy, limit, offset)
   *
   * @example
   * const docs = await db.select('documents', {
   *   columns: ['id', 'title', 'status'],
   *   where: { status: 'published' },
   *   orderBy: { column: 'createdAt', direction: 'DESC' },
   *   limit: 10,
   * });
   */
  async select<T = Record<string, unknown>>(table: string, options: SelectOptions = {}): Promise<T[]> {
    const columns = options.columns?.map((c) => this.escapeIdentifier(c)).join(', ') || '*';
    const { whereClause, params } = this.buildWhereClause(options.where);

    let sql = `SELECT ${columns} FROM ${this.escapeIdentifier(table)}${whereClause}`;

    if (options.orderBy) {
      const orderBy =
        typeof options.orderBy === 'string'
          ? options.orderBy
          : `${this.escapeIdentifier(options.orderBy.column)} ${options.orderBy.direction}`;
      sql += ` ORDER BY ${orderBy}`;
    }

    if (options.limit !== undefined) {
      sql += ` LIMIT ${options.limit}`;
    }

    if (options.offset !== undefined) {
      sql += ` OFFSET ${options.offset}`;
    }

    return this.query<T>(sql, params);
  }

  /**
   * Get a single row by ID.
   */
  async getById<T = Record<string, unknown>>(
    table: string,
    id: string | number,
    idColumn: string = 'id'
  ): Promise<T | null> {
    const sql = `SELECT * FROM ${this.escapeIdentifier(table)} WHERE ${this.escapeIdentifier(idColumn)} = ? LIMIT 1`;
    const results = await this.query<T>(sql, [id]);
    return results[0] || null;
  }

  // ─────────────────────────────────────────────────────────────────────────────
  // Private methods
  // ─────────────────────────────────────────────────────────────────────────────

  private async executeLocal<T>(sql: string, params?: SQLParam[]): Promise<T[]> {
    const result = await this.executeLocalRaw<T>(sql, params);
    return result.results;
  }

  private async executeLocalRaw<T>(sql: string, params?: SQLParam[]): Promise<D1QueryResult<T>> {
    // Build the wrangler command
    const dbName = this.config.databaseName;

    // Escape the SQL for shell
    const escapedSql = sql.replace(/'/g, "'\\''");

    // For parameterized queries, we need to substitute the params
    // Note: This is a simplified approach. For production, consider using --json flag
    let finalSql = escapedSql;
    if (params && params.length > 0) {
      let paramIndex = 0;
      finalSql = escapedSql.replace(/\?/g, () => {
        const param = params[paramIndex++];
        if (param === null) return 'NULL';
        if (typeof param === 'string') return `'${param.replace(/'/g, "''")}'`;
        if (typeof param === 'boolean') return param ? '1' : '0';
        return String(param);
      });
    }

    try {
      const output = execSync(`npx wrangler d1 execute ${dbName} --local --json --command='${finalSql}'`, {
        encoding: 'utf-8',
        maxBuffer: 50 * 1024 * 1024, // 50MB buffer for large results
      });

      // Parse JSON output
      const parsed = JSON.parse(output);

      // Wrangler returns an array of results
      if (Array.isArray(parsed) && parsed.length > 0) {
        return parsed[0] as D1QueryResult<T>;
      }

      return {
        success: true,
        results: [],
        meta: {},
      };
    } catch (error) {
      const message = error instanceof Error ? error.message : String(error);
      throw new Error(`Local D1 query failed: ${message}`);
    }
  }

  private async executeRemote<T>(sql: string, params?: SQLParam[]): Promise<T[]> {
    const result = await this.executeRemoteRaw<T>(sql, params);
    return result.results;
  }

  private async executeRemoteRaw<T>(sql: string, params?: SQLParam[]): Promise<D1QueryResult<T>> {
    validateRemoteConfig(this.config);

    const { accountId, databaseId, apiToken } = this.config;
    const url = `https://api.cloudflare.com/client/v4/accounts/${accountId}/d1/database/${databaseId}/query`;

    const response = await fetch(url, {
      method: 'POST',
      headers: {
        Authorization: `Bearer ${apiToken}`,
        'Content-Type': 'application/json',
      },
      body: JSON.stringify({
        sql,
        params: params || [],
      }),
    });

    if (!response.ok) {
      const text = await response.text();
      throw new Error(`D1 API error (${response.status}): ${text}`);
    }

    const data = (await response.json()) as D1APIResponse<T>;

    if (!data.success) {
      const errorMessages = data.errors.map((e) => e.message).join(', ');
      throw new Error(`D1 query failed: ${errorMessages}`);
    }

    // D1 REST API returns array of results, we want the first one
    return data.result[0] || { success: true, results: [], meta: {} };
  }

  private buildWhereClause(where?: Record<string, unknown>): { whereClause: string; params: SQLParam[] } {
    if (!where || Object.keys(where).length === 0) {
      return { whereClause: '', params: [] };
    }

    const conditions: string[] = [];
    const params: SQLParam[] = [];

    for (const [key, value] of Object.entries(where)) {
      if (value === null) {
        conditions.push(`${this.escapeIdentifier(key)} IS NULL`);
      } else if (Array.isArray(value)) {
        const placeholders = value.map(() => '?').join(', ');
        conditions.push(`${this.escapeIdentifier(key)} IN (${placeholders})`);
        params.push(...(value as SQLParam[]));
      } else {
        conditions.push(`${this.escapeIdentifier(key)} = ?`);
        params.push(value as SQLParam);
      }
    }

    return {
      whereClause: ` WHERE ${conditions.join(' AND ')}`,
      params,
    };
  }

  private escapeIdentifier(identifier: string): string {
    // SQLite uses double quotes for identifiers
    return `"${identifier.replace(/"/g, '""')}"`;
  }
}

// Default singleton instance
export const db = new DBClient();
