/**
 * ts-agent-sdk Database Configuration
 *
 * Environment variables for D1 database access:
 * - SDK_DB_MODE: 'local' | 'remote' | 'auto' (default: 'auto')
 * - SDK_D1_DATABASE_NAME: D1 database binding name from wrangler.jsonc
 * - SDK_D1_DATABASE_ID: D1 database ID (required for remote mode)
 * - SDK_CF_ACCOUNT_ID: Cloudflare account ID (required for remote mode)
 * - SDK_CF_API_TOKEN: Cloudflare API token (required for remote mode)
 */

import { existsSync } from 'fs';

export type DBMode = 'local' | 'remote' | 'auto';

export interface DBConfig {
  mode: DBMode;
  databaseName: string;
  databaseId: string;
  accountId: string;
  apiToken: string;
}

/**
 * Load database configuration from environment variables.
 */
export function loadDBConfig(): DBConfig {
  return {
    mode: (process.env.SDK_DB_MODE as DBMode) || 'auto',
    databaseName: process.env.SDK_D1_DATABASE_NAME || 'DB',
    databaseId: process.env.SDK_D1_DATABASE_ID || '',
    accountId: process.env.SDK_CF_ACCOUNT_ID || '',
    apiToken: process.env.SDK_CF_API_TOKEN || '',
  };
}

/**
 * Resolve the effective mode (auto → local or remote).
 *
 * Auto-detection logic:
 * - If wrangler.jsonc exists in current directory → local
 * - Otherwise → remote
 */
export function resolveDBMode(config: DBConfig): 'local' | 'remote' {
  if (config.mode === 'auto') {
    // Check for wrangler config in current directory or parent
    const hasWranglerConfig =
      existsSync('./wrangler.jsonc') ||
      existsSync('./wrangler.toml') ||
      existsSync('./wrangler.json');

    return hasWranglerConfig ? 'local' : 'remote';
  }
  return config.mode;
}

/**
 * Validate configuration for remote mode.
 */
export function validateRemoteConfig(config: DBConfig): void {
  if (!config.databaseId) {
    throw new Error(
      'SDK_D1_DATABASE_ID is required for remote database access. ' +
        'Find it in Cloudflare dashboard or wrangler.jsonc.'
    );
  }
  if (!config.accountId) {
    throw new Error(
      'SDK_CF_ACCOUNT_ID is required for remote database access. ' +
        'Find it in Cloudflare dashboard.'
    );
  }
  if (!config.apiToken) {
    throw new Error(
      'SDK_CF_API_TOKEN is required for remote database access. ' +
        'Create one at https://dash.cloudflare.com/profile/api-tokens'
    );
  }
}

/**
 * Get the wrangler config path if it exists.
 */
export function getWranglerConfigPath(): string | null {
  const paths = ['./wrangler.jsonc', './wrangler.toml', './wrangler.json'];
  for (const path of paths) {
    if (existsSync(path)) return path;
  }
  return null;
}
