/**
 * ts-agent-sdk Error Classes
 *
 * Typed errors for consistent error handling across SDK operations.
 * Each error has a code property for programmatic handling.
 */

/**
 * Base SDK error class. All SDK errors extend this.
 */
export class SDKError extends Error {
  code: string;

  constructor(message: string, code: string) {
    super(message);
    this.name = 'SDKError';
    this.code = code;
    // Maintains proper stack trace in V8 environments
    if (Error.captureStackTrace) {
      Error.captureStackTrace(this, this.constructor);
    }
  }
}

/**
 * Authentication failed (401).
 * Check SDK_API_TOKEN is set and valid.
 */
export class AuthError extends SDKError {
  constructor(message: string = 'Authentication failed. Check your API token.') {
    super(message, 'AUTH_ERROR');
    this.name = 'AuthError';
  }
}

/**
 * Input validation failed.
 * The request was rejected due to invalid input data.
 */
export class ValidationError extends SDKError {
  field?: string;

  constructor(message: string, field?: string) {
    super(message, 'VALIDATION_ERROR');
    this.name = 'ValidationError';
    this.field = field;
  }
}

/**
 * Resource not found (404).
 * The requested resource doesn't exist or you don't have access.
 */
export class NotFoundError extends SDKError {
  resourceType: string;
  resourceId: string;

  constructor(resourceType: string, resourceId: string) {
    super(`${resourceType} not found: ${resourceId}`, 'NOT_FOUND');
    this.name = 'NotFoundError';
    this.resourceType = resourceType;
    this.resourceId = resourceId;
  }
}

/**
 * Rate limited (429).
 * Too many requests. Wait before retrying.
 */
export class RateLimitError extends SDKError {
  retryAfterSeconds: number;

  constructor(retryAfterSeconds: number = 60) {
    super(`Rate limited. Retry after ${retryAfterSeconds} seconds.`, 'RATE_LIMIT');
    this.name = 'RateLimitError';
    this.retryAfterSeconds = retryAfterSeconds;
  }
}

/**
 * MCP protocol error.
 * The MCP server returned an error response.
 */
export class MCPError extends SDKError {
  mcpCode?: number;

  constructor(message: string, mcpCode?: number) {
    super(message, 'MCP_ERROR');
    this.name = 'MCPError';
    this.mcpCode = mcpCode;
  }
}

/**
 * Network or connection error.
 * Could not reach the server.
 */
export class NetworkError extends SDKError {
  cause?: Error;

  constructor(message: string = 'Network error. Could not reach server.', cause?: Error) {
    super(message, 'NETWORK_ERROR');
    this.name = 'NetworkError';
    this.cause = cause;
  }
}
