---
name: just-cli
description: This skill should be used when the user asks to "create a justfile", "write just recipes", "configure just settings", "add just modules", "use just attributes", "set up task automation", mentions justfile, just command runner, or task automation with just.
---

# Just Command Runner

## Overview

Expert guidance for Just, a command runner with syntax inspired by make. Use this skill for creating justfiles, writing recipes, configuring settings, and implementing task automation workflows.

## Justfile Discovery Protocol

**When a justfile exists in a project, agents should prefer just recipes over raw commands.**

### Discovery Steps

1. Check for `justfile` or `Justfile` in project root
2. Run `just --list` to discover available recipes
3. Map common operations to recipes:

| Operation   | Just Recipe        | Instead Of                          |
| ----------- | ------------------ | ----------------------------------- |
| Run tests   | `just test`        | `npm test`, `cargo test`, `pytest`  |
| Build       | `just build`       | `npm run build`, `cargo build`      |
| Lint        | `just lint`        | `npm run lint`, `cargo clippy`      |
| Format      | `just fmt`         | `npm run format`, `cargo fmt`       |
| Check       | `just check`       | `npm run check`, `cargo check`      |
| Coverage    | `just coverage`    | `npm run coverage`, `cargo llvm-cov`|
| Audit       | `just audit`       | `npm audit`, `cargo audit`          |
| Dev server  | `just dev`         | `npm run dev`, `cargo run`          |
| Clean       | `just clean`       | `rm -rf dist`, `cargo clean`        |

### Common Recipe Naming Conventions

- **test** / **t** - Run test suite
- **build** / **b** - Build project
- **lint** - Run linter
- **fmt** - Format code
- **check** - Type checking / static analysis
- **coverage** - Generate coverage report
- **audit** - Security audit
- **dev** / **serve** - Start development server
- **clean** - Remove build artifacts
- **full-check** / **fc** - Run all checks

**Key capabilities:**

- Create and organize justfiles with proper structure
- Write recipes with attributes, dependencies, and parameters
- Configure settings for shell, modules, and imports
- Use built-in constants for terminal formatting
- Implement check/write patterns for code quality tools

## Quick Reference

### Essential Settings

```just
set allow-duplicate-recipes       # Allow recipes to override imported ones
set allow-duplicate-variables     # Allow variables to override imported ones
set shell := ["bash", "-euo", "pipefail", "-c"]  # Strict bash with error handling
set unstable                      # Enable unstable features (modules, script attribute)
```

> **Environment Variables**: Use [mise](https://mise.jdx.dev/) for environment variable management instead of Just's dotenv settings. mise provides project-scoped env vars via `.mise.toml` with better tooling integration.

### Common Attributes

| Attribute                 | Purpose                                       |
| ------------------------- | --------------------------------------------- |
| `[group("name")]`         | Group recipes in `just --list` output         |
| `[no-cd]`                 | Don't change to justfile directory            |
| `[private]`               | Hide from `just --list` (same as `_` prefix)  |
| `[script]`                | Execute recipe as single script block         |
| `[script("interpreter")]` | Use specific interpreter (bash, python, etc.) |
| `[confirm("prompt")]`     | Require user confirmation before running      |
| `[doc("text")]`           | Override recipe documentation                 |
| `[positional-arguments]`  | Enable positional args for this recipe only   |

Multiple attributes can be combined:

```just
[no-cd, private]
[group("checks")]
recipe:
    echo "hello"
```

### Built-in Constants

Terminal formatting constants are globally available (no definition needed):

| Constant                                           | Description                                |
| -------------------------------------------------- | ------------------------------------------ |
| `CYAN`, `GREEN`, `RED`, `YELLOW`, `BLUE`, `PURPLE` | Text colors                                |
| `BOLD`, `ITALIC`, `UNDERLINE`, `STRIKETHROUGH`     | Text styles                                |
| `NORMAL`                                           | Reset formatting                           |
| `BG_*`                                             | Background colors (BG_RED, BG_GREEN, etc.) |
| `HEX`, `HEXLOWER`                                  | Hexadecimal digits                         |

Usage:

```just
@status:
    echo -e '{{ GREEN }}Success!{{ NORMAL }}'
    echo -e '{{ BOLD + CYAN }}Building...{{ NORMAL }}'
```

### Key Functions

```just
# Require executable exists (fails recipe if not found)
jq := require("jq")

# Get environment variable with default
log_level := env("LOG_LEVEL", "info")

# Get justfile directory path
root := justfile_dir()
```

## Recipe Patterns

### Status Reporter Pattern

Display formatted status during multi-step workflows:

```just
@_run-with-status recipe *args:
    echo ""
    echo -e '{{ CYAN }}→ Running {{ recipe }}...{{ NORMAL }}'
    just {{ recipe }} {{ args }}
    echo -e '{{ GREEN }}✓ {{ recipe }} completed{{ NORMAL }}'
alias rws := _run-with-status
```

### Check/Write Pattern

Pair check (verify) and write (fix) recipes for code quality tools:

```just
[group("checks")]
@biome-check +globs=".":
    na biome check {{ globs }}
alias bc := biome-check

[group("checks")]
@biome-write +globs=".":
    na biome check --write {{ globs }}
alias bw := biome-write
```

### Full Check/Write Pattern

Aggregate all checks with status reporting:

```just
[group("checks")]
@full-check:
    just _run-with-status biome-check
    just _run-with-status prettier-check
    just _run-with-status tsc-check
    echo ""
    echo -e '{{ GREEN }}All code checks passed!{{ NORMAL }}'
alias fc := full-check

[group("checks")]
@full-write:
    just _run-with-status biome-write
    just _run-with-status prettier-write
    echo ""
    echo -e '{{ GREEN }}All code fixes applied!{{ NORMAL }}'
alias fw := full-write
```

### Standard Alias Conventions

| Recipe         | Alias | Recipe         | Alias |
| -------------- | ----- | -------------- | ----- |
| full-check     | fc    | full-write     | fw    |
| biome-check    | bc    | biome-write    | bw    |
| prettier-check | pc    | prettier-write | pw    |
| mdformat-check | mc    | mdformat-write | mw    |
| tsc-check      | tc    | ruff-check     | rc    |
| test           | t     | build          | b     |

## Modules & Imports

### Import Pattern

Include recipes from another file:

```just
import "./just/settings.just"
import "./just/base.just"
import? "./local.just"    # Optional (no error if missing)
```

### Module Pattern

Load submodule (requires `set unstable`):

```just
mod foo                   # Loads foo.just or foo/justfile
mod bar "path/to/bar"     # Custom path
mod? optional             # Optional module

# Call module recipes
just foo::build
```

### Devkit Import Pattern

For projects using `@sablier/devkit`:

```just
import "./node_modules/@sablier/devkit/just/base.just"
import "./node_modules/@sablier/devkit/just/npm.just"
```

## Section Organization

Standard section header format:

```just
# ---------------------------------------------------------------------------- #
#                                 DEPENDENCIES                                 #
# ---------------------------------------------------------------------------- #
```

Common sections (in order):

1. **DEPENDENCIES** - Required tools with URLs
1. **CONSTANTS** - Glob patterns, environment vars
1. **RECIPES / COMMANDS** - Main entry points
1. **CHECKS** - Code quality recipes
1. **UTILITIES / INTERNAL HELPERS** - Private helpers

## Default Recipe

Always define a default recipe:

```just
# Show available commands
default:
    @just --list
```

## Dependencies Declaration

Document required tools at the top:

```just
# ---------------------------------------------------------------------------- #
#                                 DEPENDENCIES                                 #
# ---------------------------------------------------------------------------- #

# Bun: https://bun.sh
bun := require("bun")

# Ni: https://github.com/antfu-collective/ni
na := require("na")
ni := require("ni")
nlx := require("nlx")
```

## Context7 Fallback

For Just features not covered in this skill (new attributes, advanced functions, edge cases), fetch the latest documentation:

```
Use context7 MCP with library ID `/websites/just_systems-man` to get up-to-date Just documentation.
```

Example topics to search:

- `modules import mod` - Module system details
- `settings` - All available settings
- `attributes` - Recipe attributes
- `functions` - Built-in functions
- `script recipes` - Script block syntax

## Additional Resources

### Reference Files

For detailed patterns and comprehensive coverage, consult:

- **[`reference.md`](references/reference.md)** - Complete settings, modules, attributes, functions reference
- **[`patterns.md`](references/patterns.md)** - Established conventions, section organization, helper patterns

### External Documentation

- **Official Manual**: https://just.systems/man/en/
- **GitHub Repository**: https://github.com/casey/just
- **Context7 Library ID**: `/websites/just_systems-man`

## Important Limitations

**Just does not support flags.** Recipe parameters are positional only:

```just
# WRONG: Flags don't work
build --release:  # This won't work!

# CORRECT: Use positional parameters
build mode="debug":
    cargo build {{ if mode == "release" { "--release" } else { "" } }}

# Usage: just build release
```

For complex flag-like behavior, use environment variables or positional parameters with defaults.

## Tips

1. Use `@` prefix to suppress command echo: `@echo "quiet"`
1. Use `+` for variadic parameters: `test +args`
1. Use `*` for optional variadic: `build *flags`
1. Quote glob patterns in variables: `GLOBS := "\"**/*.json\""`
1. Use `[no-cd]` in monorepos to stay in current directory
1. Private recipes start with `_` or use `[private]`
1. Always define aliases after recipe names for discoverability
