# TanStack Solid Examples

## 1. TanStack Start (Server Function)
Defining and consuming a server-side function with type safety.

```tsx
// app/data/users.ts
import { createServerFn } from "@tanstack/start";

export const getUser = createServerFn({ method: "GET" })
  .validator((data: { id: string }) => data)
  .handler(async ({ data }) => {
    // This runs on the server
    return db.user.findUnique({ where: { id: data.id } });
  });

// app/routes/users/$id.tsx
import { createFileRoute } from "@tanstack/solid-router";
import { getUser } from "~/data/users";

export const Route = createFileRoute("/users/$id")({
  loader: ({ params }) => getUser({ data: { id: params.id } }),
  component: UserComponent,
});

function UserComponent() {
  const user = Route.useLoaderData();
  return <h1>{user.name}</h1>;
}
```

## 2. TanStack Query (Data Fetching)
Handling async data with reactivity. Note the use of a function for options.

```tsx
import { createQuery } from "@tanstack/solid-query";
import { createSignal, Suspense } from "solid-js";

function PostViewer(props: { postId: number }) {
  // 1. Pass a function to createQuery to track reactivity
  const query = createQuery(() => ({
    queryKey: ["post", props.postId],
    queryFn: async () => {
      const res = await fetch(\`/api/posts/\${props.postId}\`);
      return res.json();
    },
    enabled: !!props.postId, // Reactive boolean
  }));

  return (
    <div>
      <Suspense fallback="Loading...">
        {/* 2. Access properties directly, NO destructuring */}
        <h3>{query.data?.title}</h3>
        <p>{query.data?.body}</p>
        
        {/* 3. Handle errors */}
        {query.isError && <p>Error: {query.error.message}</p>}
      </Suspense>
    </div>
  );
}
```

## 3. TanStack Router (Configuration)
Setting up a router with nested routes.

```tsx
// routes/__root.tsx
import { createRootRoute, Link, Outlet } from "@tanstack/solid-router";

export const Route = createRootRoute({
  component: () => (
    <>
      <nav>
        <Link to="/" activeProps={{ class: "font-bold" }}>Home</Link>
        <Link to="/about">About</Link>
      </nav>
      <hr />
      <Outlet />
    </>
  ),
});

// app.tsx
import { RouterProvider, createRouter } from "@tanstack/solid-router";
import { routeTree } from "./routeTree.gen"; // Generated by CLI

const router = createRouter({ routeTree });

export default function App() {
  return <RouterProvider router={router} />;
}
```

## 4. TanStack Table (Headless Table)
Rendering a table with sorting and reactivity.

```tsx
import { 
  createSolidTable, 
  flexRender, 
  getCoreRowModel, 
  getSortedRowModel,
  ColumnDef
} from "@tanstack/solid-table";
import { createSignal, For } from "solid-js";

type Person = { firstName: string; age: number };

const columns: ColumnDef<Person>[] = [
  { header: "Name", accessorKey: "firstName" },
  { header: "Age", accessorKey: "age" },
];

function PeopleTable() {
  const [data] = createSignal<Person[]>([
    { firstName: "Tanner", age: 33 },
    { firstName: "Kevin", age: 31 },
  ]);
  const [sorting, setSorting] = createSignal([]);

  const table = createSolidTable({
    // Use getters for reactive data/state
    get data() { return data() },
    columns,
    state: {
      get sorting() { return sorting() },
    },
    onSortingChange: setSorting,
    getCoreRowModel: getCoreRowModel(),
    getSortedRowModel: getSortedRowModel(),
  });

  return (
    <table>
      <thead>
        <For each={table.getHeaderGroups()}>
          {(headerGroup) => (
            <tr>
              <For each={headerGroup.headers}>
                {(header) => (
                  <th onClick={header.column.getToggleSortingHandler()}>
                    {flexRender(header.column.columnDef.header, header.getContext())}
                    {{ asc: " 🔼", desc: " 🔽" }[header.column.getIsSorted() as string] ?? null}
                  </th>
                )}
              </For>
            </tr>
          )}
        </For>
      </thead>
      <tbody>
        <For each={table.getRowModel().rows}>
          {(row) => (
            <tr>
              <For each={row.getVisibleCells()}>
                {(cell) => (
                  <td>{flexRender(cell.column.columnDef.cell, cell.getContext())}</td>
                )}
              </For>
            </tr>
          )}
        </For>
      </tbody>
    </table>
  );
}
```

## 5. TanStack Form (Form State)
Headless form validation and submission.

```tsx
import { createForm } from "@tanstack/solid-form";

function SignUpForm() {
  const form = createForm(() => ({
    defaultValues: {
      email: "",
    },
    onSubmit: async ({ value }) => {
      console.log("Submitting:", value);
    },
  }));

  return (
    <form
      onSubmit={(e) => {
        e.preventDefault();
        e.stopPropagation();
        form.handleSubmit();
      }}
    >
      <form.Field
        name="email"
        validators={{
          onChange: ({ value }) => 
            !value.includes("@") ? "Invalid email" : undefined,
        }}
      >
        {(field) => (
          <div>
            <label>Email:</label>
            <input
              name={field().name}
              value={field().state.value}
              onBlur={field().handleBlur}
              onInput={(e) => field().handleChange(e.target.value)}
            />
            {field().state.meta.errors ? (
              <em role="alert">{field().state.meta.errors.join(", ")}</em>
            ) : null}
          </div>
        )}
      </form.Field>
      <button type="submit">Submit</button>
    </form>
  );
}
```
