---
name: vscode-theme
description: Generate VSCode color themes, package them as .vsix files, and install to VSCode. Use when creating custom VSCode themes, designing color schemes, or when the user mentions VSCode theme, color theme, dark theme, or light theme.
---

# VSCode Theme Generator

Generate custom VSCode color themes from scratch, package them, and install them directly to VSCode.

## Prerequisites

The following tools are required:
- `@vscode/vsce` (VSCode Extension Manager)
- `code` command (VSCode CLI)

Check if tools are installed:
```bash
which vsce || echo "vsce is not installed"
which code || echo "code command is not installed"
```

Install if missing:
```bash
# vsce
npm install -g @vscode/vsce

# code command: In VSCode, press Cmd+Shift+P → "Shell Command: Install 'code' command in PATH"
```

## Theme Builder

This skill includes a theme builder script that enables efficient theme creation by editing individual part files.

### Directory Structure

```
<theme-id>/
├── .vscodeignore          # Excludes parts/ from .vsix
├── package.json           # Extension manifest
├── parts/                 # Edit these files to customize
│   ├── base.json          # name, type, semanticHighlighting
│   ├── colors-editor.json # Editor colors
│   ├── colors-ui.json     # UI colors (sidebar, tabs, etc.)
│   ├── colors-terminal.json # Terminal colors
│   ├── tokens.json        # Syntax highlighting
│   └── semantic.json      # Semantic token colors
└── themes/
    └── <theme-id>-color-theme.json  # Generated by merge
```

### Builder Commands

```bash
# Initialize a new theme from template
node <skill-path>/theme-builder.js init <theme-id> "<theme-name>" [--type dark|light]

# Merge parts into final theme file
node <skill-path>/theme-builder.js merge <theme-id>

# Package as .vsix
node <skill-path>/theme-builder.js package <theme-id>

# Bump version
node <skill-path>/theme-builder.js bump <theme-id> [patch|minor|major]
```

## Instructions

### Step 1: Gather theme requirements

Ask the user for:
- Theme name (e.g., "Ocean Blue")
- Theme type: dark or light
- Color preferences (base colors, accent colors)
- Any specific style inspiration

### Step 2: Initialize theme

```bash
node <skill-path>/theme-builder.js init <theme-id> "<Theme Name>" --type dark
```

Replace:
- `<skill-path>`: Path to this skill directory
- `<theme-id>`: lowercase with hyphens (e.g., `ocean-blue`)
- `<Theme Name>`: display name (e.g., `Ocean Blue`)

### Step 3: Edit part files

Edit the files in `<theme-id>/parts/` to customize the theme.

#### parts/base.json
```json
{
  "name": "Theme Name",
  "type": "dark",
  "semanticHighlighting": true
}
```

#### parts/colors-editor.json
Editor-related colors:

| Property | Description |
|----------|-------------|
| `editor.background` | Main editor background |
| `editor.foreground` | Default text color |
| `editorLineNumber.foreground` | Line number color |
| `editorCursor.foreground` | Cursor color |
| `editor.selectionBackground` | Selection highlight |
| `editorBracketMatch.*` | Matching bracket highlight |
| `editorError.foreground` | Error underline |
| `editorWarning.foreground` | Warning underline |

#### parts/colors-ui.json
UI element colors:

| Property | Description |
|----------|-------------|
| `activityBar.background` | Left sidebar icon bar |
| `sideBar.background` | File explorer background |
| `statusBar.background` | Bottom status bar |
| `titleBar.activeBackground` | Window title bar |
| `tab.activeBackground` | Active tab |
| `tab.inactiveBackground` | Inactive tabs |
| `list.activeSelectionBackground` | Selected item in lists |
| `input.background` | Input field background |
| `button.background` | Button background |

#### parts/colors-terminal.json
Terminal colors including ANSI colors:

| Property | Description |
|----------|-------------|
| `terminal.background` | Terminal background |
| `terminal.foreground` | Terminal text |
| `terminal.ansiBlack` - `terminal.ansiWhite` | ANSI colors |
| `terminal.ansiBrightBlack` - `terminal.ansiBrightWhite` | Bright ANSI colors |

#### parts/tokens.json
Syntax highlighting (array of token rules):

| Scope | Applies to |
|-------|-----------|
| `comment` | Comments |
| `string` | String literals |
| `keyword` | `if`, `for`, `const`, etc. |
| `storage.type` | Type keywords |
| `entity.name.function` | Function names |
| `entity.name.type` | Type/class names |
| `variable` | Variables |
| `constant.numeric` | Numbers |
| `entity.name.tag` | HTML/XML tags |
| `entity.other.attribute-name` | Attributes |

Example token rule:
```json
{
  "scope": ["comment", "punctuation.definition.comment"],
  "settings": { "foreground": "#6A9955", "fontStyle": "italic" }
}
```

#### parts/semantic.json
Semantic token colors (optional, object format):
```json
{
  "function": "#DCDCAA",
  "variable.readonly": "#4FC1FF"
}
```

### Step 4: Merge and package

```bash
# Merge parts into theme file
node <skill-path>/theme-builder.js merge <theme-id>

# Package as .vsix
node <skill-path>/theme-builder.js package <theme-id>

# Install to VSCode
code --install-extension <theme-id>/<theme-id>-0.0.1.vsix
```

### Step 5: Notify user

After installation:
> Theme installed successfully!
> Press `Cmd+K Cmd+T` (Mac) or `Ctrl+K Ctrl+T` (Windows/Linux) to select your new theme.

## Updating an Existing Theme

When updating a theme, bump the version before packaging:

```bash
# Bump version (default: patch)
node <skill-path>/theme-builder.js bump <theme-id>

# Merge and package
node <skill-path>/theme-builder.js merge <theme-id>
node <skill-path>/theme-builder.js package <theme-id>

# Install updated theme
code --install-extension <theme-id>/<theme-id>-<new-version>.vsix
```

## Color Design Tips

### Dark themes
- Background: `#1a1a2e` to `#2d2d44`
- Foreground: `#d4d4d4` to `#eaeaea`
- Limit accent colors to 1-2

### Light themes
- Background: `#ffffff` to `#f5f5f5`
- Foreground: `#1a1a1a` to `#333333`
- Use muted accent colors

### Transparency
Use alpha channel for subtle effects: `#ffffff40` (last two digits = opacity 00-ff)

### Accessibility
Aim for reasonable contrast between text and background for readability.

## Troubleshooting

### Theme not appearing
Run `Developer: Reload Window` from command palette.
