# Complex Build Workflow

For building complete features requiring multiple shadcn components.

## Phase 1: Requirements Analysis

**Input:** User request for a complete feature/section

**Steps:**

1. Call `shadcn___get_project_registries` - Note all available registries

2. Break down the request into components needed:
   ```
   Example: "login form" needs:
   - Form (validation)
   - Input (email, password)
   - Button (submit)
   - Label (field labels)
   - Card (container)
   - Alert (error messages)
   ```

3. For each identified component:
   - Call `shadcn___search_items_in_registries`
   - Verify it exists in registry
   - Note exact name

4. **Output** component hierarchy:
   ```
   ## Feature: [Name]
   
   ## Components Required:
   - form (validation and submission)
   - input (email and password fields)
   - button (submit action)
   - card (form container)
   - alert (error display)
   
   ## Component Hierarchy:
   Card
   └── Form
       ├── Label + Input (email)
       ├── Label + Input (password)
       ├── Button (submit)
       └── Alert (errors)
   ```

## Phase 2: Component Research

**Input:** Component list from Phase 1

**Steps:**

1. For each component:
   
   a. Get implementation details:
   ```
   shadcn___view_items_in_registries(items: ["@shadcn/component"])
   ```
   - Note file dependencies
   - Note key props
   
   b. Get examples:
   ```
   shadcn___get_item_examples_from_registries(registries, query: "component-demo")
   ```
   - For forms: get validation examples
   - For data: get loading state examples

2. Get installation command for ALL components at once:
   ```
   shadcn___get_add_command_for_items(items: ["@shadcn/form", "@shadcn/input", ...])
   ```

3. **Output** research summary with:
   - Installation commands
   - Key imports for each component
   - Relevant example code snippets
   - Important props to use

## Phase 3: Implementation

**Input:** Requirements + Research from previous phases

**Steps:**

1. Build implementation following:
   - Use EXACT imports from research
   - Follow hierarchy from requirements
   - Adapt examples to match use case
   - Add proper TypeScript types
   - Include state management (useState, form hooks)
   - Add error handling

2. Run audit:
   ```
   shadcn___get_audit_checklist
   ```
   - Verify best practices followed

3. **Output** complete implementation:
   ```tsx
   // All necessary imports
   import { Form, FormControl, FormField } from "@/components/ui/form"
   import { Input } from "@/components/ui/input"
   import { Button } from "@/components/ui/button"
   
   export function FeatureName() {
     // Full implementation
   }
   ```

4. Include setup instructions:
   - Installation commands needed
   - Where to add the component
   - Any additional setup (providers, configs)

## Example: Login Form

**Phase 1 Output:**
```
Components: card, form, input, button, label, alert
Hierarchy: Card > Form > (Label+Input)*2 + Button + Alert
```

**Phase 2 Output:**
```bash
npx shadcn@latest add card form input button label alert
```

**Phase 3 Output:**
```tsx
"use client"

import { zodResolver } from "@hookform/resolvers/zod"
import { useForm } from "react-hook-form"
import * as z from "zod"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form"
import { Input } from "@/components/ui/input"

const formSchema = z.object({
  email: z.string().email(),
  password: z.string().min(8),
})

export function LoginForm() {
  const form = useForm<z.infer<typeof formSchema>>({
    resolver: zodResolver(formSchema),
  })

  function onSubmit(values: z.infer<typeof formSchema>) {
    console.log(values)
  }

  return (
    <Card>
      <CardHeader>
        <CardTitle>Login</CardTitle>
      </CardHeader>
      <CardContent>
        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
            <FormField
              control={form.control}
              name="email"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>Email</FormLabel>
                  <FormControl>
                    <Input placeholder="email@example.com" {...field} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <FormField
              control={form.control}
              name="password"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>Password</FormLabel>
                  <FormControl>
                    <Input type="password" {...field} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
            <Button type="submit" className="w-full">Login</Button>
          </form>
        </Form>
      </CardContent>
    </Card>
  )
}
```
