# RSC Data Fetching Patterns

## Table of Contents
- [Basic SSR Fetch](#basic-ssr-fetch)
- [Parallel Data Fetching](#parallel-data-fetching)
- [Hybrid Pattern](#hybrid-pattern)
- [Streaming with Suspense](#streaming-with-suspense)
- [Error Handling](#error-handling)
- [Caching Strategies](#caching-strategies)
- [Common Anti-Patterns](#common-anti-patterns)

---

## Basic SSR Fetch

Simple server-side data fetching:

```tsx
// app/products/page.tsx
import { getProducts } from "@/lib/actions/products";

export default async function ProductsPage() {
  const products = await getProducts();
  
  return (
    <div>
      {products.map(product => (
        <ProductCard key={product.id} product={product} />
      ))}
    </div>
  );
}
```

---

## Parallel Data Fetching

Fetch multiple data sources simultaneously:

```tsx
// app/dashboard/page.tsx
export default async function DashboardPage() {
  // ✅ Parallel fetching - both run at same time
  const [users, orders, stats] = await Promise.all([
    getUsers(),
    getOrders(),
    getStats(),
  ]);
  
  return (
    <div>
      <UsersSection users={users} />
      <OrdersSection orders={orders} />
      <StatsSection stats={stats} />
    </div>
  );
}
```

**Avoid sequential fetching:**

```tsx
// ❌ Bad - waterfall, each waits for previous
const users = await getUsers();
const orders = await getOrders();
const stats = await getStats();
```

---

## Hybrid Pattern

Server-side initial data + client-side interactivity:

```tsx
// app/products/page.tsx (Server Component)
import { getProducts, getCategories } from "@/lib/actions";
import { ProductsClient } from "./products-client";

export default async function ProductsPage() {
  const [products, categories] = await Promise.all([
    getProducts(),
    getCategories(),
  ]);
  
  return (
    <ProductsClient 
      initialProducts={products} 
      categories={categories} 
    />
  );
}

// app/products/products-client.tsx (Client Component)
"use client";

import { useState } from "react";

interface ProductsClientProps {
  initialProducts: Product[];
  categories: Category[];
}

export function ProductsClient({ initialProducts, categories }: ProductsClientProps) {
  const [products, setProducts] = useState(initialProducts);
  const [selectedCategory, setSelectedCategory] = useState<string | null>(null);
  
  const handleFilter = async (categoryId: string) => {
    setSelectedCategory(categoryId);
    // Client-side fetch for filtering
    const filtered = await filterProducts(categoryId);
    setProducts(filtered);
  };
  
  return (
    <div>
      <CategoryFilter 
        categories={categories} 
        onFilter={handleFilter} 
      />
      <ProductGrid products={products} />
    </div>
  );
}
```

---

## Streaming with Suspense

Show content progressively as data loads:

```tsx
// app/dashboard/page.tsx
import { Suspense } from "react";

export default function DashboardPage() {
  return (
    <div>
      {/* Shows immediately */}
      <h1>Dashboard</h1>
      
      {/* Streams in when ready */}
      <Suspense fallback={<StatsSkeleton />}>
        <StatsSection />
      </Suspense>
      
      <Suspense fallback={<ChartSkeleton />}>
        <ChartSection />
      </Suspense>
    </div>
  );
}

// Async component - fetches its own data
async function StatsSection() {
  const stats = await getStats(); // Can be slow
  return <Stats data={stats} />;
}
```

---

## Error Handling

Handle fetch errors gracefully:

```tsx
// app/products/page.tsx
import { getProducts } from "@/lib/actions/products";

export default async function ProductsPage() {
  const products = await getProducts();
  
  if (!products || products.length === 0) {
    return <EmptyState message="No products found" />;
  }
  
  return <ProductList products={products} />;
}

// With error boundary (error.tsx)
// app/products/error.tsx
"use client";

export default function Error({
  error,
  reset,
}: {
  error: Error;
  reset: () => void;
}) {
  return (
    <div>
      <h2>Something went wrong!</h2>
      <button onClick={() => reset()}>Try again</button>
    </div>
  );
}
```

---

## Caching Strategies

### Static Data (Build Time)

```tsx
// Data fetched at build time, cached indefinitely
export const dynamic = "force-static";

export default async function Page() {
  const data = await getData();
  return <Content data={data} />;
}
```

### Revalidate on Interval

```tsx
// Revalidate every 60 seconds
export const revalidate = 60;

export default async function Page() {
  const data = await getData();
  return <Content data={data} />;
}
```

### On-Demand Revalidation

```tsx
// lib/actions/products.ts
"use server";

import { revalidatePath } from "next/cache";

export async function createProduct(data: ProductInput) {
  await db.products.create(data);
  revalidatePath("/products"); // Invalidate cache
}
```

### No Cache (Always Fresh)

```tsx
export const dynamic = "force-dynamic";

export default async function Page() {
  const data = await getData(); // Always fresh
  return <Content data={data} />;
}
```

---

## Common Anti-Patterns

### ❌ Client-side fetch for static data

```tsx
"use client";
export default function Page() {
  const [data, setData] = useState([]);
  useEffect(() => {
    fetch("/api/data").then(r => r.json()).then(setData);
  }, []);
  return <List data={data} />;
}
```

**Fix:** Remove `"use client"`, use async function.

### ❌ Fetching in layout for page-specific data

```tsx
// app/layout.tsx
export default async function Layout({ children }) {
  const user = await getUser(); // Fetched on every page
  return <div>{children}</div>;
}
```

**Fix:** Fetch in page component or use React cache().

### ❌ Over-fetching with Context

```tsx
// Fetches ALL data even if page only needs users
const { users, products, orders } = useStore();
```

**Fix:** Fetch only what each page needs server-side.

### ❌ Ignoring parallel fetching

```tsx
const a = await fetchA();
const b = await fetchB(); // Waits for A to complete
```

**Fix:** Use `Promise.all([fetchA(), fetchB()])`.
