# Neatoo Data Mapping Reference

## Overview

Data mapping in Neatoo transfers property values between domain entities and persistence entities. This is done through explicit property assignments in factory methods, giving you full control over data flow between rich domain objects and flat EF Core database entities.

## Mapper Methods Overview

<!-- snippet: overview -->
```csharp
/// <summary>
/// Person entity demonstrating MapFrom, MapTo, MapModifiedTo.
/// </summary>
public partial interface IPersonWithMapper : IEntityBase
{
    int Id { get; }
    string? FirstName { get; set; }
    string? LastName { get; set; }
    string? Email { get; set; }

    void MapFrom(PersonEntity entity);
    void MapTo(PersonEntity entity);
}

[Factory]
internal partial class PersonWithMapper : EntityBase<PersonWithMapper>, IPersonWithMapper
{
    public PersonWithMapper(IEntityBaseServices<PersonWithMapper> services) : base(services) { }

    public partial int Id { get; set; }
    public partial string? FirstName { get; set; }
    public partial string? LastName { get; set; }
    public partial string? Email { get; set; }

    // Manually implemented mappers
    public void MapFrom(PersonEntity entity)
    {
        this.Id = entity.Id;
        this.FirstName = entity.FirstName;
        this.LastName = entity.LastName;
        this.Email = entity.Email;
    }

    public void MapTo(PersonEntity entity)
    {
        entity.Id = this.Id;
        entity.FirstName = this.FirstName;
        entity.LastName = this.LastName;
        entity.Email = this.Email;
    }

    // Generated by Neatoo BaseGenerator - checks IsModified for each property
    public partial void MapModifiedTo(PersonEntity entity);

    [Create]
    public void Create()
    {
        // New person with no ID yet
    }

    [Fetch]
    public void Fetch(PersonEntity entity)
    {
        MapFrom(entity);
    }
}
```
<!-- /snippet -->

## MapFrom - Loading from Database

Use in `[Fetch]` operations to load data from persistence entities.

<!-- snippet: map-from -->
```csharp
/// <summary>
/// Demonstrates MapFrom usage in Fetch operations.
/// </summary>
public partial interface IFetchableProduct : IEntityBase
{
    int Id { get; }
    string? Name { get; set; }
    decimal Price { get; set; }
    int StockQuantity { get; set; }

    void MapFrom(ProductEntity entity);
}

public class ProductEntity
{
    public int Id { get; set; }
    public string? Name { get; set; }
    public decimal Price { get; set; }
    public int StockQuantity { get; set; }
}

[Factory]
internal partial class FetchableProduct : EntityBase<FetchableProduct>, IFetchableProduct
{
    public FetchableProduct(IEntityBaseServices<FetchableProduct> services) : base(services) { }

    public partial int Id { get; set; }
    public partial string? Name { get; set; }
    public partial decimal Price { get; set; }
    public partial int StockQuantity { get; set; }

    public void MapFrom(ProductEntity entity)
    {
        this.Id = entity.Id;
        this.Name = entity.Name;
        this.Price = entity.Price;
        this.StockQuantity = entity.StockQuantity;
    }

    [Create]
    public void Create() { }

    /// <summary>
    /// Fetch operation uses MapFrom to load entity data.
    /// During Fetch, rules are paused via FactoryStart().
    /// </summary>
    [Fetch]
    public void Fetch(ProductEntity entity)
    {
        MapFrom(entity);  // Copy all properties
    }
}
```
<!-- /snippet -->

## MapTo - Saving to Database

Use in `[Insert]` operations to create new persistence entities.

<!-- snippet: map-to -->
```csharp
/// <summary>
/// Demonstrates MapTo usage in Insert operations.
/// </summary>
public partial interface IInsertableItem : IEntityBase
{
    int Id { get; }
    string? Description { get; set; }
    decimal Amount { get; set; }

    void MapTo(ItemEntity entity);
}

public class ItemEntity
{
    public int Id { get; set; }
    public string? Description { get; set; }
    public decimal Amount { get; set; }
}

[Factory]
internal partial class InsertableItem : EntityBase<InsertableItem>, IInsertableItem
{
    public InsertableItem(IEntityBaseServices<InsertableItem> services) : base(services) { }

    public partial int Id { get; set; }
    public partial string? Description { get; set; }
    public partial decimal Amount { get; set; }

    public void MapTo(ItemEntity entity)
    {
        entity.Description = this.Description;
        entity.Amount = this.Amount;
        // Note: ID is typically database-generated
    }

    [Create]
    public void Create() { }

    /// <summary>
    /// Insert uses MapTo to copy all properties to new entity.
    /// </summary>
    [Insert]
    public Task Insert()
    {
        // Simulate insert
        var entity = new ItemEntity();
        MapTo(entity);

        // Simulate database-generated ID
        entity.Id = 1;
        this.Id = entity.Id;

        return Task.CompletedTask;
    }
}
```
<!-- /snippet -->

## MapModifiedTo - Updating Database

Use in `[Update]` operations to only update changed properties.

<!-- snippet: map-modified-to -->
```csharp
/// <summary>
/// Demonstrates MapModifiedTo for efficient updates.
/// </summary>
public partial interface IUpdatableRecord : IEntityBase
{
    int Id { get; }
    string? Title { get; set; }
    string? Content { get; set; }
    DateTime? LastModified { get; set; }

    void MapFrom(RecordEntity entity);
}

public class RecordEntity
{
    public int Id { get; set; }
    public string? Title { get; set; }
    public string? Content { get; set; }
    public DateTime? LastModified { get; set; }
}

[Factory]
internal partial class UpdatableRecord : EntityBase<UpdatableRecord>, IUpdatableRecord
{
    public UpdatableRecord(IEntityBaseServices<UpdatableRecord> services) : base(services) { }

    public partial int Id { get; set; }
    public partial string? Title { get; set; }
    public partial string? Content { get; set; }
    public partial DateTime? LastModified { get; set; }

    public void MapFrom(RecordEntity entity)
    {
        this.Id = entity.Id;
        this.Title = entity.Title;
        this.Content = entity.Content;
        this.LastModified = entity.LastModified;
    }

    // MapModifiedTo is source-generated - only updates IsModified properties
    public partial void MapModifiedTo(RecordEntity entity);

    [Create]
    public void Create() { }

    [Fetch]
    public void Fetch(RecordEntity entity)
    {
        MapFrom(entity);
    }

    /// <summary>
    /// Update uses MapModifiedTo for efficiency.
    /// Only modified properties are copied to the entity.
    /// </summary>
    [Update]
    public Task Update(RecordEntity entity)
    {
        MapModifiedTo(entity);  // Only changed properties
        return Task.CompletedTask;
    }
}
```
<!-- /snippet -->

## Custom Mapping Logic

For complex scenarios with transformations or value objects.

<!-- snippet: custom-mapping -->
```csharp
/// <summary>
/// Demonstrates custom mapping logic for computed fields.
/// </summary>
public partial interface IEmployeeWithComputed : IEntityBase
{
    int Id { get; }
    string? FirstName { get; set; }
    string? LastName { get; set; }
    string? FullName { get; }
    string? PhoneType { get; set; }
}

public class EmployeeEntity
{
    public int Id { get; set; }
    public string? FirstName { get; set; }
    public string? LastName { get; set; }
    public string? PhoneTypeString { get; set; }
}

[Factory]
internal partial class EmployeeWithComputed : EntityBase<EmployeeWithComputed>, IEmployeeWithComputed
{
    public EmployeeWithComputed(IEntityBaseServices<EmployeeWithComputed> services) : base(services) { }

    public partial int Id { get; set; }
    public partial string? FirstName { get; set; }
    public partial string? LastName { get; set; }

    // Computed property - not partial, excluded from mappers
    public string? FullName => $"{FirstName} {LastName}";

    public partial string? PhoneType { get; set; }

    [Create]
    public void Create() { }

    /// <summary>
    /// Custom mapping with computed fields and type conversions.
    /// </summary>
    [Fetch]
    public void Fetch(EmployeeEntity entity)
    {
        this.Id = entity.Id;
        this.FirstName = entity.FirstName;
        this.LastName = entity.LastName;

        // Custom mapping for type conversion
        this.PhoneType = entity.PhoneTypeString;
    }
}
```
<!-- /snippet -->

## Different Shapes

Mapping between domain and persistence entities with different structures.

<!-- snippet: different-shapes -->
```csharp
/// <summary>
/// Demonstrates mapping between different entity structures.
/// Domain model has flat address fields, EF has nested Address object.
/// </summary>
public partial interface ICustomerWithAddress : IEntityBase
{
    int Id { get; }
    string? FirstName { get; set; }
    string? LastName { get; set; }
    string? StreetAddress { get; set; }
    string? City { get; set; }
    string? State { get; set; }
    string? ZipCode { get; set; }

    void MapFromEntity(CustomerEntity entity);
    void MapToEntity(CustomerEntity entity);
}

[Factory]
internal partial class CustomerWithAddress : EntityBase<CustomerWithAddress>, ICustomerWithAddress
{
    public CustomerWithAddress(IEntityBaseServices<CustomerWithAddress> services) : base(services) { }

    public partial int Id { get; set; }
    public partial string? FirstName { get; set; }
    public partial string? LastName { get; set; }
    public partial string? StreetAddress { get; set; }
    public partial string? City { get; set; }
    public partial string? State { get; set; }
    public partial string? ZipCode { get; set; }

    /// <summary>
    /// Custom mapping from nested EF structure to flat domain model.
    /// </summary>
    public void MapFromEntity(CustomerEntity entity)
    {
        Id = entity.Id;
        FirstName = entity.FirstName;
        LastName = entity.LastName;
        // EF has nested Address object
        StreetAddress = entity.Address?.Street;
        City = entity.Address?.City;
        State = entity.Address?.State;
        ZipCode = entity.Address?.ZipCode;
    }

    /// <summary>
    /// Custom mapping from flat domain model to nested EF structure.
    /// </summary>
    public void MapToEntity(CustomerEntity entity)
    {
        entity.Id = Id;
        entity.FirstName = FirstName;
        entity.LastName = LastName;
        entity.Address ??= new AddressEntity();
        entity.Address.Street = StreetAddress;
        entity.Address.City = City;
        entity.Address.State = State;
        entity.Address.ZipCode = ZipCode;
    }

    [Create]
    public void Create() { }

    [Fetch]
    public void Fetch(CustomerEntity entity)
    {
        MapFromEntity(entity);
    }

    [Insert]
    public Task Insert()
    {
        var entity = new CustomerEntity();
        MapToEntity(entity);
        // Simulate insert...
        return Task.CompletedTask;
    }
}
```
<!-- /snippet -->

## EF Core Integration

### DbContext Interface

<!-- snippet: dbcontext-interface -->
```csharp
public interface ISampleDbContext
{
    DbSet<PersonEntity> Persons { get; }
    Task<PersonEntity?> FindPerson(Guid id);
    void AddPerson(PersonEntity person);
    Task<int> SaveChangesAsync(CancellationToken cancellationToken = default);
}
```
<!-- /snippet -->

### DbContext Class

<!-- snippet: dbcontext-class -->
```csharp
public class SampleDbContext : DbContext, ISampleDbContext
{
    public virtual DbSet<PersonEntity> Persons { get; set; } = null!;

    public string DbPath { get; }

    public SampleDbContext()
    {
        var folder = Environment.SpecialFolder.LocalApplicationData;
        var path = Environment.GetFolderPath(folder);
        DbPath = Path.Join(path, "NeatooSamples.db");
    }

    protected override void OnConfiguring(DbContextOptionsBuilder optionsBuilder)
        => optionsBuilder.UseSqlite($"Data Source={DbPath}")
                         .UseLazyLoadingProxies();

    protected override void OnModelCreating(ModelBuilder modelBuilder)
    {
        base.OnModelCreating(modelBuilder);
        modelBuilder.Entity<PersonEntity>().Property(e => e.Id).ValueGeneratedNever();
    }

    public void AddPerson(PersonEntity person) => Persons.Add(person);

    public Task<PersonEntity?> FindPerson(Guid id)
        => Persons.FirstOrDefaultAsync(p => p.Id == id);
}
```
<!-- /snippet -->

### Entity Class

<!-- snippet: entity-class -->
```csharp
public class PersonEntity
{
    [Key]
    public Guid Id { get; set; }

    [Required]
    public string FirstName { get; set; } = null!;

    [Required]
    public string LastName { get; set; } = null!;

    public string? Email { get; set; }
}
```
<!-- /snippet -->

## Key Concepts

1. **LoadProperty vs Setter** - Use `LoadProperty` in Fetch to avoid triggering rules
2. **IsModified check** - Only update properties that changed
3. **Value Objects** - Flatten to columns or use JSON serialization
4. **Explicit mapping** - No magic, full control over data flow

## Best Practices

1. **Keep persistence entities simple** - Flat structures map to database tables
2. **Use LoadProperty for loading** - Prevents rule execution during fetch
3. **Check IsModified before update** - Reduces unnecessary database writes
4. **Map value objects explicitly** - Break down into primitive columns
