# Ralph Wiggum Loop

> Autonomous AI coding with clean slate iterations, PRD-driven features, and CI quality gates.

## Quick Start

```bash
# Edit prd.json to define your features
vim prd.json

# Run the loop
./ralph-loop.sh

# Monitor progress
tail -f progress.txt
```

## How It Works

```
┌─────────────────────────────────────────────┐
│  ITERATION 1                                 │
│  ┌─────────────────────────────────────┐    │
│  │ 1. Read PRD + progress.txt          │    │
│  │ 2. Pick highest priority incomplete │    │
│  │ 3. Implement feature                │    │
│  │ 4. Run tests (must pass)            │    │
│  │ 5. Run typecheck (must pass)        │    │
│  │ 6. Commit code                      │    │
│  │ 7. Update prd.json (passes: true)   │    │
│  │ 8. Append to progress.txt           │    │
│  │ 9. Check if all done?               │    │
│  │    → Yes: Exit with COMPLETE        │    │
│  │    → No: Continue to iteration 2    │    │
│  └─────────────────────────────────────┘    │
└─────────────────────────────────────────────┘
                      ↓
┌─────────────────────────────────────────────┐
│  ITERATION 2 (clean slate)                  │
│  ┌─────────────────────────────────────┐    │
│  │ Fresh context, repeat above...      │    │
│  └─────────────────────────────────────┘    │
└─────────────────────────────────────────────┘
```

**Key insight:** Each iteration has a clean slate (fresh context), but maintains "memory" through:
- Git history (code changes)
- progress.txt (what was done)
- prd.json (what's left to do)

## Files

| File | Purpose |
|------|---------|
| `ralph-loop.sh` | Main loop script (executable) |
| `prd.json` | User stories to implement |
| `progress.txt` | Progress log (appended each iteration) |

## PRD Best Practices

### Good User Story

```json
{
  "id": "US-001",
  "priority": 1,
  "title": "Add date formatting utility",
  "description": "Create a utility function that formats Date objects",
  "acceptanceCriteria": [
    "File src/utils/date.ts exists",
    "Function formatDateISO(date: Date): string exists",
    "Unit tests exist with 5+ test cases",
    "All tests pass"
  ],
  "passes": false
}
```

**Why it's good:**
- Specific and scoped (one function)
- Clear acceptance criteria (testable)
- Completable in one iteration

### Bad User Story

```json
{
  "id": "US-001",
  "title": "Build the frontend",
  "acceptanceCriteria": ["UI looks good", "Everything works"]
}
```

**Why it's bad:**
- Too vague and broad
- Criteria not testable
- Too large for one iteration

## Configuration

Edit `ralph-loop.sh` to customize:

```bash
MAX_ITERATIONS=10           # Max loops (prevent infinite)
PRD_FILE="prd.json"         # PRD location
PROGRESS_FILE="progress.txt" # Progress log location
```

## Monitoring

```bash
# Watch progress in real-time
tail -f progress.txt

# Watch git commits
watch -n 5 'git log --oneline -10'

# Check PRD status
cat prd.json | grep -A 1 '"passes"'
```

## Troubleshooting

| Problem | Solution |
|---------|----------|
| Loop doesn't complete anything | Check test commands in ralph-loop.sh |
| Agent picks wrong priority | Verify priority numbers (1 = highest) |
| Tests keep failing | Run tests manually first |
| Reached max iterations | Increase MAX_ITERATIONS or break down stories |

## Tips

1. **Start small** - Test with 2-3 simple stories first
2. **Watch iteration 1** - Catch issues early
3. **Specific criteria** - Use file paths and function names
4. **Keep CI healthy** - Ensure tests are fast and reliable

---

Generated by Ralph Wiggum Loop Skill
