#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
DesignSystem-Pro-Max Audit - Audit CSS for design system compliance
Usage: python audit.py <file.css> --stack <stack>

Checks:
- Naming conventions (kebab-case, CSS variables)
- Property value validity (colors, spacing tokens)
- Accessibility (color contrast, focus states)
- Best practices (shorthands, specificity)
"""

import argparse
import re
import sys
from pathlib import Path
from collections import defaultdict

# Add parent directory to path for imports
sys.path.insert(0, str(Path(__file__).parent))

from core import (
    DOMAIN_CONFIG, STACK_CONFIG, AVAILABLE_STACKS,
    search, search_stack, get_all_specs
)


# Audit rule definitions
AUDIT_RULES = {
    "naming": {
        "camel_case": {
            "pattern": r'\.[a-z][a-zA-Z0-9]*[A-Z]',
            "severity": "medium",
            "message": "Class names should use kebab-case, not camelCase",
            "fix": "Convert to kebab-case: .myClass -> .my-class"
        },
        "snake_case": {
            "pattern": r'\.[a-z][a-z0-9_]*_[a-z0-9_]+',
            "severity": "low",
            "message": "Class names should use kebab-case, not snake_case",
            "fix": "Convert to kebab-case: .my_class -> .my-class"
        },
        "css_var_notation": {
            "pattern": r'--[a-zA-Z0-9_]*[A-Z]',
            "severity": "low",
            "message": "CSS variables should use kebab-case",
            "fix": "Convert to kebab-case: --myVar -> --my-var"
        }
    },
    "values": {
        "hardcoded_color": {
            "pattern": r'(color|background|border-color):\s*(#[0-9a-fA-F]{3,6}|rgb|hsl)\b',
            "severity": "high",
            "message": "Use CSS variables/tokens instead of hardcoded colors",
            "fix": "Replace with CSS variable: color: #1890ff -> color: var(--primary-color)"
        },
        "hardcoded_spacing": {
            "pattern": r'(margin|padding):\s*\d+px\s*(?:!important)?;',
            "severity": "medium",
            "message": "Use spacing tokens instead of hardcoded pixel values",
            "fix": "Replace with spacing token: padding: 16px -> padding: var(--spacing-md)"
        },
        "hardcoded_size": {
            "pattern": r'(font-size|border-radius|width|height):\s*\d+px\s*(?:!important)?;',
            "severity": "medium",
            "message": "Use design tokens for consistent sizing",
            "fix": "Replace with size token: font-size: 14px -> font-size: var(--font-size-base)"
        },
        "important": {
            "pattern": r'!important',
            "severity": "medium",
            "message": "Avoid !important - use specificity or better selectors",
            "fix": "Remove !important and restructure selectors"
        }
    },
    "accessibility": {
        "no_focus_state": {
            "pattern": r'\.[a-z][a-z0-9-]*\s*\{[^}]*\}(?!\s*\.[a-z][a-z0-9-]*:focus)',
            "severity": "high",
            "message": "Interactive elements should have :focus states defined",
            "fix": "Add :focus styles for keyboard navigation"
        },
        "low_contrast_warning": {
            "pattern": r'color:\s*#(?:f0f0f0|e0e0e0|d0d0d0|c0c0c0|b0b0b0)',
            "severity": "high",
            "message": "Light gray text may have insufficient contrast",
            "fix": "Use darker color for WCAG AA compliance (4.5:1 ratio)"
        },
        "pointer_missing": {
            "pattern": r'\.(button|btn|clickable|interactive)\s*\{[^}]*\}(?![^}]*cursor:\s*pointer)',
            "severity": "medium",
            "message": "Clickable elements should have cursor: pointer",
            "fix": "Add: cursor: pointer"
        }
    },
    "best_practices": {
        "missing_vendor_prefix": {
            "pattern": r'(box-sizing|border-radius|transform|transition|box-shadow)\s*:',
            "severity": "low",
            "message": "Consider modern CSS - vendor prefixes usually not needed",
            "fix": "Remove unnecessary -webkit-, -moz- prefixes (use Autoprefixer if needed)"
        },
        "universal_selector": {
            "pattern": r'\*\s*\{',
            "severity": "medium",
            "message": "Universal selector * can impact performance",
            "fix": "Use specific selectors or reset.css instead"
        },
        "deep_selector": {
            "pattern": r'\.[a-z][a-z0-9-]*\s*\.[a-z][a-z0-9-]*\s*\.[a-z][a-z0-9-]*\s*\.[a-z][a-z0-9-]*',
            "severity": "low",
            "message": "Deep selectors (4+ levels) are hard to maintain",
            "fix": "Flatten selector structure or use BEM methodology"
        },
        "id_selector": {
            "pattern": r'#[a-zA-Z][a-zA-Z0-9_-]*\s*\{',
            "severity": "medium",
            "message": "Avoid ID selectors - use classes for reusability",
            "fix": "Replace #id with .class for better reusability"
        }
    }
}


class CSSAuditor:
    """CSS code auditor for design system compliance"""

    def __init__(self, stack):
        self.stack = stack
        self.violations = []
        self.fixes = []
        self.stats = {
            "by_category": defaultdict(int),
            "by_severity": defaultdict(int)
        }

        # Load stack-specific rules
        self.stack_rules = self._load_stack_rules()

    def _load_stack_rules(self):
        """Load stack-specific compliance rules"""
        rules = get_all_specs(stack=self.stack)

        stack_specific = {
            "framework_guidelines": [],
            "component_patterns": {},
            "required_tokens": set(),
            "forbidden_patterns": []
        }

        for rule in rules:
            if not rule or not isinstance(rule, dict):
                continue
            category = rule.get("Category", "")
            guideline = rule.get("Guideline", "")
            description = rule.get("Description", "")
            severity_val = rule.get("Severity", "medium")
            severity = str(severity_val).lower() if severity_val else "medium"
            code_good = rule.get("Code Good", "")
            code_bad = rule.get("Code Bad", "")

            if category and guideline:
                stack_specific["framework_guidelines"].append({
                    "category": category,
                    "guideline": guideline,
                    "description": description,
                    "severity": severity,
                    "good": code_good,
                    "bad": code_bad
                })

        return stack_specific

    def audit_file(self, file_path):
        """Audit CSS file for compliance violations"""
        try:
            with open(file_path, 'r', encoding='utf-8') as f:
                content = f.read()
        except Exception as e:
            self.violations.append({
                "category": "file_error",
                "severity": "critical",
                "message": f"Cannot read file: {e}",
                "line": 0,
                "fix": "Check file path and permissions"
            })
            return self.violations

        lines = content.split('\n')

        # Run all audit rules
        self._check_naming_conventions(lines)
        self._check_property_values(lines)
        self._check_accessibility(lines)
        self._check_best_practices(lines)
        self._check_stack_compliance(content)

        return self.violations

    def _check_naming_conventions(self, lines):
        """Check naming convention compliance"""
        for i, line in enumerate(lines, 1):
            for rule_name, rule in AUDIT_RULES["naming"].items():
                pattern = rule["pattern"]
                if re.search(pattern, line):
                    self.violations.append({
                        "category": "naming",
                        "rule": rule_name,
                        "severity": rule["severity"],
                        "message": rule["message"],
                        "line": i,
                        "content": line.strip(),
                        "fix": rule["fix"]
                    })
                    self.stats["by_category"]["naming"] += 1
                    self.stats["by_severity"][rule["severity"]] += 1

    def _check_property_values(self, lines):
        """Check property value compliance"""
        for i, line in enumerate(lines, 1):
            for rule_name, rule in AUDIT_RULES["values"].items():
                pattern = rule["pattern"]
                if re.search(pattern, line):
                    self.violations.append({
                        "category": "values",
                        "rule": rule_name,
                        "severity": rule["severity"],
                        "message": rule["message"],
                        "line": i,
                        "content": line.strip(),
                        "fix": rule["fix"]
                    })
                    self.stats["by_category"]["values"] += 1
                    self.stats["by_severity"][rule["severity"]] += 1

    def _check_accessibility(self, lines):
        """Check accessibility compliance"""
        for i, line in enumerate(lines, 1):
            for rule_name, rule in AUDIT_RULES["accessibility"].items():
                # Skip complex regex patterns that need multi-line analysis
                if rule_name == "no_focus_state":
                    continue

                pattern = rule["pattern"]
                if re.search(pattern, line):
                    self.violations.append({
                        "category": "accessibility",
                        "rule": rule_name,
                        "severity": rule["severity"],
                        "message": rule["message"],
                        "line": i,
                        "content": line.strip(),
                        "fix": rule["fix"]
                    })
                    self.stats["by_category"]["accessibility"] += 1
                    self.stats["by_severity"][rule["severity"]] += 1

        # Check for missing :focus states (multi-line)
        content = '\n'.join(lines)
        class_pattern = r'\.([a-z][a-z0-9-]*)\s*\{([^}]*)\}'
        classes = re.finditer(class_pattern, content, re.MULTILINE | re.DOTALL)

        for match in classes:
            class_name = match.group(1)
            class_content = match.group(2)

            # Check if this might be an interactive class
            if any(kw in class_name for kw in ["button", "btn", "click", "interactive", "link", "nav"]):
                # Look for :focus definition
                focus_pattern = rf'\.{class_name}:focus\s*\{{'
                if not re.search(focus_pattern, content):
                    # Find approximate line number
                    line_num = content[:match.start()].count('\n') + 1
                    self.violations.append({
                        "category": "accessibility",
                        "rule": "missing_focus_state",
                        "severity": "high",
                        "message": f"Interactive class '.{class_name}' missing :focus state",
                        "line": line_num,
                        "content": f".{class_name} {{ ... }}",
                        "fix": f"Add .{class_name}:focus {{ outline: 2px solid var(--color-focus); }}"
                    })
                    self.stats["by_category"]["accessibility"] += 1
                    self.stats["by_severity"]["high"] += 1

    def _check_best_practices(self, lines):
        """Check best practices compliance"""
        for i, line in enumerate(lines, 1):
            for rule_name, rule in AUDIT_RULES["best_practices"].items():
                pattern = rule["pattern"]
                if re.search(pattern, line):
                    self.violations.append({
                        "category": "best_practices",
                        "rule": rule_name,
                        "severity": rule["severity"],
                        "message": rule["message"],
                        "line": i,
                        "content": line.strip(),
                        "fix": rule["fix"]
                    })
                    self.stats["by_category"]["best_practices"] += 1
                    self.stats["by_severity"][rule["severity"]] += 1

    def _check_stack_compliance(self, content):
        """Check framework-specific compliance"""
        for guideline in self.stack_rules["framework_guidelines"]:
            # Check for bad patterns
            bad_pattern = guideline.get("bad", "")
            if bad_pattern:
                # Escape special regex characters
                escaped_bad = re.escape(bad_pattern)
                if re.search(escaped_bad, content):
                    # Find line number
                    line_num = content[:content.find(bad_pattern)].count('\n') + 1
                    self.violations.append({
                        "category": "stack_compliance",
                        "rule": guideline["guideline"],
                        "severity": guideline["severity"],
                        "message": guideline["description"],
                        "line": line_num,
                        "content": bad_pattern,
                        "fix": f"Use: {guideline['good']}"
                    })
                    self.stats["by_category"]["stack_compliance"] += 1
                    self.stats["by_severity"][guideline["severity"]] += 1

    def get_summary(self):
        """Get audit summary"""
        return {
            "total_violations": len(self.violations),
            "by_category": dict(self.stats["by_category"]),
            "by_severity": dict(self.stats["by_severity"]),
            "stack": self.stack
        }


def format_report(auditor, format="text"):
    """Format audit report"""
    summary = auditor.get_summary()
    violations = auditor.violations

    if format == "json":
        import json
        return json.dumps({
            "summary": summary,
            "violations": violations
        }, indent=2)

    # Text format
    lines = []
    lines.append("=" * 60)
    lines.append("DesignSystem Pro Max - CSS Audit Report")
    lines.append("=" * 60)
    lines.append("")
    lines.append(f"Stack: {auditor.stack}")
    lines.append(f"Total Violations: {summary['total_violations']}")
    lines.append("")

    # Summary by severity
    lines.append("Summary by Severity:")
    severity_order = ["critical", "high", "medium", "low"]
    for severity in severity_order:
        count = summary["by_severity"].get(severity, 0)
        if count > 0:
            lines.append(f"  {severity.upper()}: {count}")
    lines.append("")

    # Summary by category
    if summary["by_category"]:
        lines.append("Summary by Category:")
        for category, count in summary["by_category"].items():
            lines.append(f"  {category}: {count}")
        lines.append("")

    # Violations
    if violations:
        lines.append("Violations:")
        lines.append("")

        # Sort by severity and line number
        sorted_violations = sorted(
            violations,
            key=lambda v: (severity_order.index(v.get("severity", "low")), v.get("line", 0))
        )

        for v in sorted_violations:
            severity = v.get("severity", "low").upper()
            line = v.get("line", 0)
            category = v.get("category", "unknown")
            rule = v.get("rule", "")
            message = v.get("message", "")
            content = v.get("content", "")
            fix = v.get("fix", "")

            lines.append(f"[{severity}] Line {line} - {category}: {rule}")
            lines.append(f"  Message: {message}")
            if content:
                lines.append(f"  Content: {content[:80]}...")
            if fix:
                lines.append(f"  Fix: {fix}")
            lines.append("")

    else:
        lines.append("No violations found! CSS is compliant.")
        lines.append("")

    lines.append("=" * 60)

    return "\n".join(lines)


def main():
    parser = argparse.ArgumentParser(
        description="DesignSystem Pro Max - Audit CSS for compliance",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  python audit.py styles.css --stack react-antd
  python audit.py button.css --stack tailwind --format json

Stacks: %(stacks)s
        """ % {
            "stacks": ", ".join(AVAILABLE_STACKS)
        }
    )

    parser.add_argument(
        "file",
        help="CSS file to audit"
    )

    parser.add_argument(
        "--stack", "-s",
        choices=AVAILABLE_STACKS,
        required=True,
        help="Target framework/stack"
    )

    parser.add_argument(
        "--format", "-f",
        choices=["text", "json"],
        default="text",
        help="Output format (default: text)"
    )

    parser.add_argument(
        "--output", "-o",
        help="Output file (default: stdout)"
    )

    args = parser.parse_args()

    # Run audit
    auditor = CSSAuditor(args.stack)
    auditor.audit_file(args.file)

    # Format report
    report = format_report(auditor, args.format)

    # Output
    if args.output:
        with open(args.output, 'w', encoding='utf-8') as f:
            f.write(report)
        print(f"Audit report saved: {args.output}")
    else:
        print(report)

    # Return exit code based on violations
    summary = auditor.get_summary()
    return 1 if summary["total_violations"] > 0 else 0


if __name__ == "__main__":
    sys.exit(main())
