#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
DesignSystem-Pro-Max Generate - Generate compliant CSS code from design specifications
Usage: python generate.py "<query>" --stack <stack> --output <file.css>
"""

import argparse
import sys
from pathlib import Path

# Add parent directory to path for imports
sys.path.insert(0, str(Path(__file__).parent))

from core import (
    DOMAIN_CONFIG, STACK_CONFIG, AVAILABLE_STACKS, AVAILABLE_DOMAINS,
    search, search_stack, get_all_specs
)


# CSS Templates for different output types
CSS_TEMPLATES = {
    "variables": """/* Design System Variables - {stack} */
/* Generated by DesignSystem-Pro-Max */

{content}
""",

    "component": """/* Component Styles - {name} */
/* Framework: {framework} */

.{class_name} {{
  /* Base styles */
{styles}
}}

/* States */
.{class_name}:hover {{
  /* Hover state */
{hover_styles}
}}

.{class_name}:focus {{
  /* Focus state */
{focus_styles}
}}

.{class_name}:disabled {{
  /* Disabled state */
{disabled_styles}
}}
""",

    "color": """/* Color Tokens - {stack} */
:root {{
{css_variables}
}}
""",

    "spacing": """/* Spacing Scale - {stack} */
:root {{
{css_variables}
}}

/* Utility Classes */
{utility_classes}
""",

    "typography": """/* Typography Scale - {stack} */
:root {{
{css_variables}
}}

/* Typography Classes */
{typography_classes}
"""
}


def generate_css_from_tokens(specs, stack):
    """Generate CSS variables from token specifications"""
    if not specs:
        return ""

    lines = []
    for spec in specs:
        token = spec.get("Token", "")
        value = spec.get("Value", "")
        if token and value:
            # Format as CSS variable
            var_name = token.replace("--", "").replace("_", "-").lower()
            lines.append(f"  --{var_name}: {value};")

    return "\n".join(lines)


def generate_color_css(stack):
    """Generate color token CSS"""
    specs = get_all_specs(domain="color")

    # Filter by framework - handle react-antd -> ant design mapping
    framework_map = {
        "react-antd": "ant design",
        "material-ui": "material-ui",
        "figma-tokens": "figma",
        "bootstrap": "bootstrap",
        "tailwind": "tailwind",
        "chakra": "chakra",
        "elements": "adobe"
    }

    target_framework = framework_map.get(stack.lower(), stack.lower())

    filtered_specs = []
    for spec in specs:
        framework = spec.get("Framework", "").lower()
        if target_framework in framework or framework == "" or framework == "all":
            filtered_specs.append(spec)

    # If no specs match filter, use all specs
    if not filtered_specs:
        filtered_specs = specs

    css_vars = generate_css_from_tokens(filtered_specs, stack)

    return CSS_TEMPLATES["color"].format(
        stack=stack,
        css_variables=css_vars
    )


def generate_spacing_css(stack):
    """Generate spacing scale CSS"""
    specs = get_all_specs(domain="spacing")

    css_vars = generate_css_from_tokens(specs, stack)

    # Generate utility classes
    utility_lines = []
    for spec in specs:
        token = spec.get("Token", "")
        value = spec.get("Value", "")
        if token and value:
            # Create utility class
            class_name = token.replace("--spacing-", "").replace("_", "-")
            utility_lines.append(f".spacing-{class_name} {{ margin: {value}; padding: {value}; }}")
            utility_lines.append(f".m-{class_name} {{ margin: {value}; }}")
            utility_lines.append(f".p-{class_name} {{ padding: {value}; }}")

    return CSS_TEMPLATES["spacing"].format(
        stack=stack,
        css_variables=css_vars,
        utility_classes="\n".join(utility_lines)
    )


def generate_typography_css(stack):
    """Generate typography scale CSS"""
    specs = get_all_specs(domain="typography")

    # Generate CSS variables
    css_lines = []
    for spec in specs:
        token = spec.get("Token", "")
        if "size" in token.lower():
            value = spec.get("Size", "")
            if value:
                var_name = token.replace("--", "").replace("_", "-").lower()
                css_lines.append(f"  --{var_name}: {value};")

        if "weight" in token.lower():
            value = spec.get("Weight", "")
            if value:
                var_name = token.replace("--", "").replace("_", "-").lower()
                css_lines.append(f"  --{var_name}: {value};")

    # Generate typography classes
    class_lines = []
    for spec in specs:
        token = spec.get("Token", "")
        size = spec.get("Size", "")
        weight = spec.get("Weight", "")
        line_height = spec.get("Line Height", "")

        if size or weight or line_height:
            class_name = token.replace("--font-", "").replace("_", "-").lower()
            styles = []
            if size:
                styles.append(f"font-size: {size}")
            if weight:
                styles.append(f"font-weight: {weight}")
            if line_height:
                styles.append(f"line-height: {line_height}")

            class_lines.append(f".font-{class_name} {{ {'; '.join(styles)}; }}")

    return CSS_TEMPLATES["typography"].format(
        stack=stack,
        css_variables="\n".join(css_lines),
        typography_classes="\n".join(class_lines)
    )


def generate_component_css(query, stack):
    """Generate component CSS from search results"""
    result = search(query, domain="component", max_results=1)

    if not result.get("results"):
        return "/* No component found for query: " + query + " */"

    component_spec = result["results"][0]

    name = component_spec.get("Component", "Component")
    framework = component_spec.get("Framework", stack)
    styles = component_spec.get("Styles", "")

    # Parse styles into CSS properties
    style_lines = []
    if styles:
        # Simple style parsing (comma-separated)
        for style in styles.split(","):
            style = style.strip()
            if style:
                # Convert description to CSS
                style_lines.append(f"  /* {style} */")

    return CSS_TEMPLATES["component"].format(
        name=name,
        framework=framework,
        class_name=name.lower().replace(" ", "-"),
        styles="\n".join(style_lines) or "  /* Add component styles */",
        hover_styles="  /* Add hover styles */",
        focus_styles="  /* Add focus styles */",
        disabled_styles="  /* Add disabled styles */"
    )


def generate_from_query(query, stack, output_type="auto"):
    """Generate CSS from search query"""
    query_lower = query.lower()

    # Auto-detect output type
    if output_type == "auto":
        if any(kw in query_lower for kw in ["color", "palette", "theme"]):
            output_type = "color"
        elif any(kw in query_lower for kw in ["spacing", "margin", "padding", "gap"]):
            output_type = "spacing"
        elif any(kw in query_lower for kw in ["typography", "font", "text", "heading"]):
            output_type = "typography"
        else:
            output_type = "component"

    # Generate based on type
    if output_type == "color":
        return generate_color_css(stack)
    elif output_type == "spacing":
        return generate_spacing_css(stack)
    elif output_type == "typography":
        return generate_typography_css(stack)
    else:
        return generate_component_css(query, stack)


def main():
    parser = argparse.ArgumentParser(
        description="DesignSystem Pro Max - Generate compliant CSS code",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  python generate.py "primary color tokens" --stack react-antd --output colors.css
  python generate.py "spacing scale" --stack tailwind --output spacing.css
  python generate.py "button styles" --stack bootstrap --output button.css

Stacks: %(stacks)s
        """ % {
            "stacks": ", ".join(AVAILABLE_STACKS)
        }
    )

    parser.add_argument(
        "query",
        help="Search query (what to generate)"
    )

    parser.add_argument(
        "--stack", "-s",
        choices=AVAILABLE_STACKS,
        required=True,
        help="Target framework/stack"
    )

    parser.add_argument(
        "--output", "-o",
        required=True,
        help="Output CSS file path"
    )

    parser.add_argument(
        "--format", "-f",
        choices=["auto", "color", "spacing", "typography", "component"],
        default="auto",
        help="Output format (default: auto-detect)"
    )

    parser.add_argument(
        "--stdout",
        action="store_true",
        help="Output to stdout instead of file"
    )

    args = parser.parse_args()

    # Generate CSS
    css_content = generate_from_query(args.query, args.stack, args.format)

    # Output
    if args.stdout:
        print(css_content)
    else:
        output_path = Path(args.output)
        output_path.parent.mkdir(parents=True, exist_ok=True)
        with open(output_path, 'w', encoding='utf-8') as f:
            f.write(css_content)
        print(f"CSS generated: {output_path}")
        print(f"Stack: {args.stack}")
        print(f"Type: {args.format}")

    return 0


if __name__ == "__main__":
    sys.exit(main())
