#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
DesignSystem-Pro-Max Search - CLI entry point for searching design specifications
Usage: python search.py "<query>" [--domain <domain>] [--stack <stack>] [--max-results 3]

Domains: component, color, typography, spacing, tokens
Stacks: react-antd, material-ui, figma-tokens, bootstrap, tailwind, chakra, elements
"""

import argparse
import json
import sys

# Add parent directory to path for imports
from pathlib import Path
sys.path.insert(0, str(Path(__file__).parent))

from core import (
    DOMAIN_CONFIG, STACK_CONFIG, AVAILABLE_DOMAINS, AVAILABLE_STACKS,
    MAX_RESULTS, search, search_stack
)


def format_output(result):
    """Format results for Claude/AI consumption (token-optimized Markdown)"""
    if "error" in result:
        return f"## DesignSystem Pro Max Error\n\n**Error:** {result['error']}"

    output = []

    # Header
    if result.get("stack"):
        output.append(f"## DesignSystem Pro Max - Stack Guidelines")
        output.append(f"**Stack:** {result['stack']} | **Query:** {result['query']}")
    else:
        output.append(f"## DesignSystem Pro Max - Search Results")
        output.append(f"**Domain:** {result['domain']} | **Query:** {result['query']}")

    output.append(f"**Source:** {result['file']} | **Found:** {result['count']} results\n")

    # Results
    for i, row in enumerate(result['results'], 1):
        output.append(f"### Result {i}")
        for key, value in row.items():
            if key == "_score":
                output.append(f"- **Relevance Score:** {value}")
            else:
                value_str = str(value)
                # Truncate long values
                if len(value_str) > 300:
                    value_str = value_str[:300] + "..."
                output.append(f"- **{key}:** {value_str}")
        output.append("")

    return "\n".join(output)


def format_json(result):
    """Format results as JSON"""
    # Remove _score from JSON output for cleaner output
    clean_result = {}
    for key, value in result.items():
        if key == "results":
            clean_result[key] = [
                {k: v for k, v in r.items() if k != "_score"}
                for r in value
            ]
        else:
            clean_result[key] = value
    return json.dumps(clean_result, indent=2, ensure_ascii=False)


def main():
    parser = argparse.ArgumentParser(
        description="DesignSystem Pro Max - Search design specifications",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  python search.py "AntD primary button" --domain component
  python search.py "color tokens" --stack figma-tokens
  python search.py "spacing" --domain spacing --max-results 5

Domains: %(domains)s
Stacks: %(stacks)s
        """ % {
            "domains": ", ".join(AVAILABLE_DOMAINS),
            "stacks": ", ".join(AVAILABLE_STACKS)
        }
    )

    parser.add_argument(
        "query",
        help="Search query (keywords to search for)"
    )

    parser.add_argument(
        "--domain", "-d",
        choices=AVAILABLE_DOMAINS,
        help="Search domain (if not specified, auto-detected)"
    )

    parser.add_argument(
        "--stack", "-s",
        choices=AVAILABLE_STACKS,
        help="Stack-specific search (takes priority over domain)"
    )

    parser.add_argument(
        "--max-results", "-n",
        type=int,
        default=MAX_RESULTS,
        help=f"Maximum results (default: {MAX_RESULTS})"
    )

    parser.add_argument(
        "--json",
        action="store_true",
        help="Output as JSON"
    )

    parser.add_argument(
        "--list-domains",
        action="store_true",
        help="List available domains and exit"
    )

    parser.add_argument(
        "--list-stacks",
        action="store_true",
        help="List available stacks and exit"
    )

    args = parser.parse_args()

    # Handle list commands
    if args.list_domains:
        print("Available domains:")
        for domain in AVAILABLE_DOMAINS:
            config = DOMAIN_CONFIG[domain]
            print(f"  - {domain}: {config['file']}")
        return 0

    if args.list_stacks:
        print("Available stacks:")
        for stack in AVAILABLE_STACKS:
            config = STACK_CONFIG[stack]
            print(f"  - {stack}: {config['file']}")
        return 0

    # Stack search takes priority
    if args.stack:
        result = search_stack(args.query, args.stack, args.max_results)
    else:
        result = search(args.query, args.domain, args.max_results)

    # Output
    if args.json:
        print(format_json(result))
    else:
        print(format_output(result))

    # Return error code if search failed
    return 1 if "error" in result else 0


if __name__ == "__main__":
    sys.exit(main())
