# pytest Test Template
# Generated by ln-743-test-infrastructure
# Location: tests/unit/test_sample.py

"""Sample test module demonstrating pytest patterns."""

import pytest


# Sample class for testing (replace with actual import)
class Calculator:
    """Sample calculator class."""

    def add(self, a: int, b: int) -> int:
        """Add two numbers."""
        return a + b

    def divide(self, a: int, b: int) -> float:
        """Divide two numbers."""
        if b == 0:
            raise ValueError("Cannot divide by zero")
        return a / b


# Fixtures - reusable test setup
@pytest.fixture
def calculator() -> Calculator:
    """Provide a Calculator instance for tests."""
    return Calculator()


class TestCalculator:
    """Test class for Calculator."""

    # Basic test
    def test_add_positive_numbers(self, calculator: Calculator) -> None:
        """Test adding positive numbers."""
        # Arrange
        a, b = 2, 3

        # Act
        result = calculator.add(a, b)

        # Assert
        assert result == 5

    # Test with expected exception
    def test_divide_by_zero_raises_error(self, calculator: Calculator) -> None:
        """Test that dividing by zero raises ValueError."""
        # Arrange & Act & Assert
        with pytest.raises(ValueError, match="Cannot divide by zero"):
            calculator.divide(10, 0)

    # Parameterized test
    @pytest.mark.parametrize(
        "a,b,expected",
        [
            (1, 1, 2),
            (0, 5, 5),
            (-1, 1, 0),
            (100, 200, 300),
        ],
    )
    def test_add_various_inputs(
        self, calculator: Calculator, a: int, b: int, expected: int
    ) -> None:
        """Test add with various input combinations."""
        assert calculator.add(a, b) == expected

    # Test with approximate comparison (for floats)
    def test_divide_returns_float(self, calculator: Calculator) -> None:
        """Test that divide returns correct float result."""
        result = calculator.divide(10, 3)
        assert result == pytest.approx(3.333, rel=0.01)


# Async test example (requires pytest-asyncio)
@pytest.mark.asyncio
async def test_async_operation() -> None:
    """Example async test."""
    import asyncio

    async def async_add(a: int, b: int) -> int:
        await asyncio.sleep(0.01)  # Simulate async operation
        return a + b

    result = await async_add(2, 3)
    assert result == 5


# Marked test (can be skipped with -m "not slow")
@pytest.mark.slow
def test_slow_operation() -> None:
    """Example of a slow test that can be skipped."""
    import time

    time.sleep(0.1)
    assert True
