// xUnit Test Template
// Generated by ln-743-test-infrastructure
// Location: tests/{Project}.Tests/SampleServiceTests.cs

using FluentAssertions;
using Moq;
using Xunit;

namespace Project.Tests;

/// <summary>
/// Sample test class demonstrating xUnit patterns.
/// </summary>
public class SampleServiceTests
{
    /// <summary>
    /// Test: Method returns expected result.
    /// Naming: MethodName_Scenario_ExpectedBehavior
    /// </summary>
    [Fact]
    public void Calculate_WithValidInput_ReturnsCorrectResult()
    {
        // Arrange
        var service = new SampleService();
        var input = 5;

        // Act
        var result = service.Calculate(input);

        // Assert
        result.Should().Be(10);
    }

    /// <summary>
    /// Test: Method throws exception for invalid input.
    /// </summary>
    [Fact]
    public void Calculate_WithNegativeInput_ThrowsArgumentException()
    {
        // Arrange
        var service = new SampleService();

        // Act
        var act = () => service.Calculate(-1);

        // Assert
        act.Should().Throw<ArgumentException>()
            .WithMessage("*negative*");
    }

    /// <summary>
    /// Test: Method uses dependency correctly.
    /// </summary>
    [Fact]
    public void Process_CallsRepositoryOnce()
    {
        // Arrange
        var mockRepository = new Mock<IRepository>();
        mockRepository.Setup(r => r.GetData()).Returns("test data");

        var service = new SampleService(mockRepository.Object);

        // Act
        service.Process();

        // Assert
        mockRepository.Verify(r => r.GetData(), Times.Once);
    }

    /// <summary>
    /// Parameterized test with Theory.
    /// </summary>
    [Theory]
    [InlineData(1, 2)]
    [InlineData(5, 10)]
    [InlineData(10, 20)]
    public void Calculate_WithVariousInputs_ReturnsDoubledValue(int input, int expected)
    {
        // Arrange
        var service = new SampleService();

        // Act
        var result = service.Calculate(input);

        // Assert
        result.Should().Be(expected);
    }
}

// Sample classes for testing (replace with actual implementations)
public class SampleService
{
    private readonly IRepository? _repository;

    public SampleService() { }

    public SampleService(IRepository repository)
    {
        _repository = repository;
    }

    public int Calculate(int input)
    {
        if (input < 0)
            throw new ArgumentException("Input cannot be negative");

        return input * 2;
    }

    public void Process()
    {
        _repository?.GetData();
    }
}

public interface IRepository
{
    string GetData();
}
