"""
Retention Analysis Core Functions

This module provides comprehensive retention analysis capabilities including
cohort analysis, churn prediction, and customer lifetime value calculation.
"""

import pandas as pd
import numpy as np
from typing import Dict, List, Optional, Tuple, Union
from sklearn.model_selection import train_test_split
from sklearn.ensemble import RandomForestClassifier, GradientBoostingClassifier
from sklearn.linear_model import LogisticRegression
from sklearn.metrics import classification_report, roc_auc_score, precision_recall_curve
from sklearn.preprocessing import StandardScaler, LabelEncoder
import warnings

class RetentionAnalyzer:
    """
    Comprehensive retention analysis toolkit.

    Provides cohort analysis, churn prediction, and CLV calculation
    for customer retention analysis.
    """

    def __init__(self):
        self.data = None
        self.cohort_matrix = None
        self.churn_model = None
        self.feature_importance = None
        self.scaler = None

    def load_data(self, data: pd.DataFrame) -> None:
        """
        Load retention analysis data.

        Args:
            data: DataFrame with user activity and retention data
        """
        self.data = data.copy()
        print(f"Loaded retention data with {len(data)} records")

    def create_cohort_matrix(self,
                           user_id_col: str,
                           date_col: str,
                           activity_col: str,
                           period_type: str = 'monthly') -> pd.DataFrame:
        """
        Create cohort retention matrix.

        Args:
            user_id_col: Column name for user identifier
            date_col: Column name for activity date
            activity_col: Column name indicating active status (1/0)
            period_type: 'monthly' or 'weekly' periods

        Returns:
            Cohort retention matrix
        """
        if self.data is None:
            raise ValueError("No data loaded. Use load_data() first.")

        # Convert date column to datetime
        self.data[date_col] = pd.to_datetime(self.data[date_col])

        # Determine period
        if period_type == 'monthly':
            self.data['period'] = self.data[date_col].dt.to_period('M')
        elif period_type == 'weekly':
            self.data['period'] = self.data[date_col].dt.to_period('W')
        else:
            raise ValueError("period_type must be 'monthly' or 'weekly'")

        # Get user's first activity period (cohort)
        cohort_periods = self.data.groupby(user_id_col)['period'].min().reset_index()
        cohort_periods.columns = [user_id_col, 'cohort_period']

        # Merge cohort information back to main data
        self.data = self.data.merge(cohort_periods, on=user_id_col)

        # Calculate period number (0 = first period)
        self.data['period_number'] = (self.data['period'] - self.data['cohort_period']).apply(lambda x: x.n)

        # Create cohort matrix
        cohort_data = self.data.groupby(['cohort_period', 'period_number'])[user_id_col].nunique().reset_index()
        cohort_sizes = self.data.groupby('cohort_period')[user_id_col].nunique().reset_index()
        cohort_sizes.columns = ['cohort_period', 'cohort_size']

        # Merge cohort sizes
        cohort_data = cohort_data.merge(cohort_sizes, on='cohort_period')

        # Calculate retention rates
        cohort_data['retention_rate'] = cohort_data[user_id_col] / cohort_data['cohort_size']

        # Create pivot table
        self.cohort_matrix = cohort_data.pivot(
            index='cohort_period',
            columns='period_number',
            values='retention_rate'
        )

        return self.cohort_matrix

    def plot_cohort_heatmap(self, figsize: Tuple[int, int] = (12, 8)) -> 'plt.Figure':
        """
        Plot cohort retention heatmap.

        Args:
            figsize: Figure size tuple

        Returns:
            Matplotlib figure object
        """
        import matplotlib.pyplot as plt
        import seaborn as sns

        if self.cohort_matrix is None:
            raise ValueError("No cohort matrix created. Use create_cohort_matrix() first.")

        fig, ax = plt.subplots(figsize=figsize)

        # Create heatmap
        sns.heatmap(self.cohort_matrix,
                   annot=True,
                   fmt='.0%',
                   cmap='YlOrRd',
                   ax=ax)

        ax.set_title('Cohort Retention Matrix', fontsize=16, fontweight='bold', pad=20)
        ax.set_xlabel('Period Number', fontsize=14)
        ax.set_ylabel('Cohort Period', fontsize=14)

        plt.tight_layout()
        return fig

    def prepare_churn_data(self,
                          target_col: str,
                          feature_cols: List[str],
                          exclude_cols: List[str] = None) -> Tuple[pd.DataFrame, pd.Series]:
        """
        Prepare data for churn prediction.

        Args:
            target_col: Target variable column (churn indicator)
            feature_cols: List of feature columns to use
            exclude_cols: Columns to exclude from features

        Returns:
            Tuple of (X_features, y_target)
        """
        if self.data is None:
            raise ValueError("No data loaded. Use load_data() first.")

        # Prepare features
        if exclude_cols:
            feature_cols = [col for col in feature_cols if col not in exclude_cols]

        X = self.data[feature_cols].copy()
        y = self.data[target_col].copy()

        # Handle categorical variables
        categorical_cols = X.select_dtypes(include=['object']).columns
        for col in categorical_cols:
            le = LabelEncoder()
            X[col] = le.fit_transform(X[col].astype(str))

        # Handle missing values
        X = X.fillna(X.median())

        self.feature_cols = feature_cols
        return X, y

    def train_churn_model(self,
                         X: pd.DataFrame,
                         y: pd.Series,
                         model_type: str = 'random_forest',
                         test_size: float = 0.2) -> Dict:
        """
        Train churn prediction model.

        Args:
            X: Feature matrix
            y: Target variable
            model_type: 'logistic', 'random_forest', 'gradient_boosting'
            test_size: Test set proportion

        Returns:
            Dictionary with model results
        """
        # Split data
        X_train, X_test, y_train, y_test = train_test_split(
            X, y, test_size=test_size, random_state=42, stratify=y
        )

        # Scale features
        self.scaler = StandardScaler()
        X_train_scaled = self.scaler.fit_transform(X_train)
        X_test_scaled = self.scaler.transform(X_test)

        # Select model
        if model_type == 'logistic':
            self.churn_model = LogisticRegression(random_state=42)
        elif model_type == 'random_forest':
            self.churn_model = RandomForestClassifier(random_state=42, n_estimators=100)
        elif model_type == 'gradient_boosting':
            self.churn_model = GradientBoostingClassifier(random_state=42, n_estimators=100)
        else:
            raise ValueError("model_type must be 'logistic', 'random_forest', or 'gradient_boosting'")

        # Train model
        if model_type == 'logistic':
            self.churn_model.fit(X_train_scaled, y_train)
        else:
            self.churn_model.fit(X_train, y_train)

        # Make predictions
        if model_type == 'logistic':
            y_pred = self.churn_model.predict(X_test_scaled)
            y_pred_proba = self.churn_model.predict_proba(X_test_scaled)[:, 1]
        else:
            y_pred = self.churn_model.predict(X_test)
            y_pred_proba = self.churn_model.predict_proba(X_test)[:, 1]

        # Calculate metrics
        results = {
            'model': self.churn_model,
            'y_true': y_test,
            'y_pred': y_pred,
            'y_pred_proba': y_pred_proba,
            'auc_score': roc_auc_score(y_test, y_pred_proba),
            'classification_report': classification_report(y_test, y_pred)
        }

        # Feature importance
        if hasattr(self.churn_model, 'feature_importances_'):
            self.feature_importance = pd.DataFrame({
                'feature': X.columns,
                'importance': self.churn_model.feature_importances_
            }).sort_values('importance', ascending=False)
        elif hasattr(self.churn_model, 'coef_'):
            self.feature_importance = pd.DataFrame({
                'feature': X.columns,
                'importance': np.abs(self.churn_model.coef_[0])
            }).sort_values('importance', ascending=False)

        return results

    def predict_churn_risk(self, X: pd.DataFrame) -> pd.DataFrame:
        """
        Predict churn risk for new data.

        Args:
            X: Feature matrix for prediction

        Returns:
            DataFrame with churn probabilities and risk categories
        """
        if self.churn_model is None:
            raise ValueError("No model trained. Use train_churn_model() first.")

        # Prepare data
        X_processed = X[self.feature_cols].copy()

        # Handle categorical variables
        categorical_cols = X_processed.select_dtypes(include=['object']).columns
        for col in categorical_cols:
            le = LabelEncoder()
            X_processed[col] = le.fit_transform(X_processed[col].astype(str))

        # Handle missing values
        X_processed = X_processed.fillna(X_processed.median())

        # Scale features if needed
        if isinstance(self.churn_model, LogisticRegression):
            X_processed = self.scaler.transform(X_processed)

        # Predict
        churn_prob = self.churn_model.predict_proba(X_processed)[:, 1]

        # Create results DataFrame
        results = X.copy()
        results['churn_probability'] = churn_prob
        results['risk_category'] = pd.cut(churn_prob,
                                        bins=[0, 0.2, 0.5, 0.8, 1.0],
                                        labels=['Low', 'Medium', 'High', 'Very High'])

        return results

    def calculate_clv(self,
                     revenue_col: str,
                     user_id_col: str,
                     time_col: str,
                     discount_rate: float = 0.1) -> pd.DataFrame:
        """
        Calculate customer lifetime value (CLV).

        Args:
            revenue_col: Column name for revenue/purchase amount
            user_id_col: Column name for user identifier
            time_col: Column name for time period
            discount_rate: Discount rate for present value calculation

        Returns:
            DataFrame with CLV calculations
        """
        if self.data is None:
            raise ValueError("No data loaded. Use load_data() first.")

        # Calculate metrics per user
        user_metrics = self.data.groupby(user_id_col).agg({
            revenue_col: ['sum', 'mean', 'count'],
            time_col: ['min', 'max']
        }).reset_index()

        # Flatten column names
        user_metrics.columns = [user_id_col, 'total_revenue', 'avg_revenue',
                               'transaction_count', 'first_period', 'last_period']

        # Calculate customer lifespan (in periods)
        user_metrics['lifespan'] = user_metrics['last_period'] - user_metrics['first_period'] + 1

        # Calculate average purchase frequency
        user_metrics['purchase_frequency'] = user_metrics['transaction_count'] / user_metrics['lifespan']

        # Calculate CLV (simplified version)
        user_metrics['clv_simple'] = user_metrics['total_revenue']

        # Calculate CLV with discounting (more sophisticated)
        user_metrics['clv_discounted'] = 0
        for _, user in user_metrics.iterrows():
            periods = user['lifespan']
            avg_revenue = user['avg_revenue']
            freq = user['purchase_frequency']

            # Present value of future cash flows
            clv = 0
            for period in range(1, int(periods) + 1):
                clv += (avg_revenue * freq) / ((1 + discount_rate) ** period)

            user_metrics.loc[user.name, 'clv_discounted'] = clv

        return user_metrics.sort_values('clv_discounted', ascending=False)

    def get_retention_summary(self) -> Dict:
        """
        Get comprehensive retention summary statistics.

        Returns:
            Dictionary with retention metrics
        """
        if self.data is None:
            raise ValueError("No data loaded. Use load_data() first.")

        summary = {
            'total_users': self.data.nunique()[self.data.columns[0]] if len(self.data.columns) > 0 else 0,
            'cohort_matrix_available': self.cohort_matrix is not None,
            'model_trained': self.churn_model is not None,
            'feature_importance_available': self.feature_importance is not None
        }

        if self.cohort_matrix is not None:
            # Cohort analysis metrics
            summary['first_period_retention'] = self.cohort_matrix.iloc[:, 0].mean()
            summary['third_period_retention'] = self.cohort_matrix.iloc[:, 2].mean() if len(self.cohort_matrix.columns) > 2 else None

        if self.feature_importance is not None:
            # Top features
            summary['top_5_features'] = self.feature_importance.head(5)['feature'].tolist()

        return summary

    def generate_retention_insights(self) -> List[str]:
        """
        Generate actionable retention insights.

        Returns:
            List of insight strings
        """
        insights = []

        if self.cohort_matrix is not None:
            # Cohort insights
            first_period_retention = self.cohort_matrix.iloc[:, 0].mean()
            insights.append(f"📊 Initial period retention: {first_period_retention:.1%}")

            if len(self.cohort_matrix.columns) > 2:
                third_period_retention = self.cohort_matrix.iloc[:, 2].mean()
                retention_drop = first_period_retention - third_period_retention
                insights.append(f"📈 Retention drop from period 1 to 3: {retention_drop:.1%}")

                if retention_drop > 0.3:
                    insights.append("⚠️ High early churn - focus on onboarding and initial engagement")

            # Cohort comparison
            if len(self.cohort_matrix) > 1:
                latest_cohort = self.cohort_matrix.iloc[-1, 0]
                earliest_cohort = self.cohort_matrix.iloc[0, 0]
                if latest_cohort > earliest_cohort:
                    insights.append("📈 Improvement in initial retention over time")
                else:
                    insights.append("📉 Decline in initial retention - investigate recent changes")

        if self.feature_importance is not None:
            # Feature importance insights
            top_features = self.feature_importance.head(3)
            insights.append("🎯 Top churn predictors:")
            for _, row in top_features.iterrows():
                insights.append(f"   • {row['feature']}")

        if self.churn_model is not None:
            insights.append("✅ Churn prediction model available for proactive retention")

        return insights

    def export_retention_report(self, filename: str = 'retention_analysis_report.html') -> str:
        """
        Export comprehensive retention analysis report.

        Args:
            filename: Output filename

        Returns:
            Path to generated report file
        """
        # Generate insights
        insights = self.generate_retention_insights()
        summary = self.get_retention_summary()

        # Create HTML report
        html_content = f"""
        <!DOCTYPE html>
        <html>
        <head>
            <title>Retention Analysis Report</title>
            <style>
                body {{ font-family: Arial, sans-serif; margin: 40px; }}
                .header {{ background-color: #f0f0f0; padding: 20px; border-radius: 5px; }}
                .metrics {{ margin: 20px 0; }}
                .insights {{ background-color: #f9f9f9; padding: 20px; border-radius: 5px; }}
                table {{ border-collapse: collapse; width: 100%; }}
                th, td {{ border: 1px solid #ddd; padding: 8px; text-align: left; }}
                th {{ background-color: #f2f2f2; }}
            </style>
        </head>
        <body>
            <div class="header">
                <h1>Retention Analysis Report</h1>
                <p>Generated on {pd.Timestamp.now().strftime('%Y-%m-%d %H:%M:%S')}</p>
            </div>

            <div class="metrics">
                <h2>Key Metrics</h2>
                <ul>
                    <li><strong>Total Users:</strong> {summary['total_users']:,}</li>
                    <li><strong>Cohort Analysis:</strong> {'Available' if summary['cohort_matrix_available'] else 'Not Available'}</li>
                    <li><strong>Churn Model:</strong> {'Trained' if summary['model_trained'] else 'Not Trained'}</li>
        """

        if 'first_period_retention' in summary:
            html_content += f"<li><strong>Initial Retention:</strong> {summary['first_period_retention']:.1%}</li>"

        if 'third_period_retention' in summary and summary['third_period_retention']:
            html_content += f"<li><strong>Third Period Retention:</strong> {summary['third_period_retention']:.1%}</li>"

        if 'top_5_features' in summary:
            html_content += f"<li><strong>Top Churn Predictors:</strong> {', '.join(summary['top_5_features'][:3])}</li>"

        html_content += """
                </ul>
            </div>

            <div class="insights">
                <h2>Key Insights</h2>
                <ul>
        """

        for insight in insights:
            html_content += f"<li>{insight}</li>"

        html_content += """
                </ul>
            </div>
        </body>
        </html>
        """

        # Write to file
        with open(filename, 'w', encoding='utf-8') as f:
            f.write(html_content)

        print(f"Report saved to {filename}")
        return filename