# Basic Patterns

Fundamental async patterns for common tasks.

## Pattern 1: Task Creation and Management

```python
import asyncio

async def background_task(name: str, delay: int):
    """Long-running background task."""
    print(f"{name} started")
    await asyncio.sleep(delay)
    print(f"{name} completed")
    return f"Result from {name}"

async def main():
    # Create tasks
    task1 = asyncio.create_task(background_task("Task 1", 2))
    task2 = asyncio.create_task(background_task("Task 2", 1))

    # Do other work
    print("Main: doing other work")
    await asyncio.sleep(0.5)

    # Wait for tasks
    result1 = await task1
    result2 = await task2

    print(f"Results: {result1}, {result2}")

asyncio.run(main())
```

## Pattern 2: Concurrent Execution with gather()

```python
import asyncio
from typing import List

async def fetch_user(user_id: int) -> dict:
    """Fetch user data."""
    await asyncio.sleep(0.5)
    return {"id": user_id, "name": f"User {user_id}"}

async def fetch_all_users(user_ids: List[int]) -> List[dict]:
    """Fetch multiple users concurrently."""
    tasks = [fetch_user(uid) for uid in user_ids]
    results = await asyncio.gather(*tasks)
    return results

async def main():
    user_ids = [1, 2, 3, 4, 5]
    users = await fetch_all_users(user_ids)
    print(f"Fetched {len(users)} users")

asyncio.run(main())
```

## Pattern 3: Sequential vs Concurrent Execution

```python
async def sequential_execution():
    """Execute tasks one after another."""
    result1 = await task1()  # Wait for task1
    result2 = await task2()  # Then wait for task2
    result3 = await task3()  # Then wait for task3
    return [result1, result2, result3]

async def concurrent_execution():
    """Execute tasks concurrently."""
    results = await asyncio.gather(
        task1(),  # All tasks start together
        task2(),
        task3()
    )
    return results
```

### Timing Comparison

```python
import asyncio
import time

async def task(n: int):
    await asyncio.sleep(1)
    return n

async def compare_timing():
    # Sequential: ~3 seconds
    start = time.time()
    result1 = await task(1)
    result2 = await task(2)
    result3 = await task(3)
    print(f"Sequential: {time.time() - start:.2f}s")

    # Concurrent: ~1 second
    start = time.time()
    results = await asyncio.gather(task(1), task(2), task(3))
    print(f"Concurrent: {time.time() - start:.2f}s")

asyncio.run(compare_timing())
```

## Pattern 4: Using as_completed()

Process results as they complete, not in order.

```python
import asyncio

async def task_with_varying_delays(n: int):
    """Task with random delay."""
    delay = 5 - n  # Task 1: 4s, Task 2: 3s, Task 3: 2s, etc.
    await asyncio.sleep(delay)
    return f"Task {n} completed (delay: {delay}s)"

async def process_as_completed():
    """Process tasks as they complete."""
    tasks = [task_with_varying_delays(i) for i in range(1, 6)]

    for coro in asyncio.as_completed(tasks):
        result = await coro
        print(result)  # Prints in completion order (5, 4, 3, 2, 1)

asyncio.run(process_as_completed())
```

## Pattern 5: Using wait()

More control over waiting for tasks.

```python
import asyncio

async def task(n: int):
    await asyncio.sleep(n)
    return f"Task {n}"

async def wait_example():
    """Wait for tasks with different strategies."""
    tasks = {asyncio.create_task(task(i)) for i in [1, 2, 3]}

    # Wait for first task to complete
    done, pending = await asyncio.wait(
        tasks,
        return_when=asyncio.FIRST_COMPLETED
    )

    for task in done:
        print(f"First completed: {task.result()}")

    # Cancel remaining
    for task in pending:
        task.cancel()

asyncio.run(wait_example())
```

### wait() Options

```python
# Wait for all tasks
done, pending = await asyncio.wait(tasks)

# Wait for first to complete
done, pending = await asyncio.wait(
    tasks,
    return_when=asyncio.FIRST_COMPLETED
)

# Wait for first exception
done, pending = await asyncio.wait(
    tasks,
    return_when=asyncio.FIRST_EXCEPTION
)

# Wait with timeout
done, pending = await asyncio.wait(
    tasks,
    timeout=5.0
)
```

## Pattern 6: Task Groups (Python 3.11+)

```python
import asyncio

async def task(n: int):
    await asyncio.sleep(1)
    return n * 2

async def task_group_example():
    """Use task groups for automatic cleanup."""
    async with asyncio.TaskGroup() as tg:
        task1 = tg.create_task(task(1))
        task2 = tg.create_task(task(2))
        task3 = tg.create_task(task(3))

    # All tasks completed here
    print(task1.result(), task2.result(), task3.result())

# Python 3.11+
# asyncio.run(task_group_example())
```

## Pattern 7: Chaining Async Operations

```python
async def fetch_user(user_id: int):
    await asyncio.sleep(0.5)
    return {"id": user_id, "name": f"User {user_id}"}

async def fetch_user_orders(user_id: int):
    await asyncio.sleep(0.5)
    return [{"id": i, "item": f"Item {i}"} for i in range(3)]

async def fetch_user_profile(user_id: int):
    await asyncio.sleep(0.5)
    return {"bio": "User bio", "location": "Earth"}

async def get_complete_user_data(user_id: int):
    """Chain async operations."""
    # Sequential - user first, then details
    user = await fetch_user(user_id)

    # Concurrent - orders and profile together
    orders, profile = await asyncio.gather(
        fetch_user_orders(user_id),
        fetch_user_profile(user_id)
    )

    return {
        **user,
        "orders": orders,
        "profile": profile
    }

asyncio.run(get_complete_user_data(1))
```

## Pattern 8: Running Synchronous Code

```python
import asyncio
import concurrent.futures

def blocking_io():
    """Blocking I/O operation."""
    import time
    time.sleep(1)
    return "Result"

async def run_blocking_in_executor():
    """Run blocking code in executor."""
    loop = asyncio.get_event_loop()

    # Run in default executor (ThreadPoolExecutor)
    result = await loop.run_in_executor(None, blocking_io)
    return result

async def run_with_custom_executor():
    """Use custom executor."""
    loop = asyncio.get_event_loop()

    with concurrent.futures.ThreadPoolExecutor(max_workers=5) as executor:
        result = await loop.run_in_executor(executor, blocking_io)
        return result

asyncio.run(run_blocking_in_executor())
```
