# Common Pitfalls

Common mistakes when working with async Python and how to avoid them.

## 1. Forgetting await

```python
# ❌ Wrong: returns coroutine object, doesn't execute
result = async_function()

# ✅ Correct
result = await async_function()
```

## 2. Blocking the Event Loop

```python
# ❌ Wrong: blocks event loop
import time
async def bad():
    time.sleep(1)  # Blocks!

# ✅ Correct
async def good():
    await asyncio.sleep(1)  # Non-blocking
```

## 3. Not Handling Cancellation

```python
# ❌ Wrong: doesn't handle cancellation
async def bad_task():
    while True:
        await asyncio.sleep(1)
        # No cleanup on cancellation

# ✅ Correct
async def good_task():
    try:
        while True:
            await asyncio.sleep(1)
    except asyncio.CancelledError:
        print("Cleaning up...")
        raise  # Re-raise
```

## 4. Mixing Sync and Async Code

```python
# ❌ Wrong: can't call async from sync directly
def sync_function():
    result = await async_function()  # SyntaxError!

# ✅ Correct
def sync_function():
    result = asyncio.run(async_function())
```

## 5. Creating Too Many Tasks

```python
# ❌ Wrong: creates all tasks at once (memory intensive)
tasks = [asyncio.create_task(process(i)) for i in range(10000)]
await asyncio.gather(*tasks)

# ✅ Correct: use semaphore to limit concurrency
semaphore = asyncio.Semaphore(100)

async def limited_process(item):
    async with semaphore:
        return await process(item)

tasks = [limited_process(i) for i in range(10000)]
await asyncio.gather(*tasks)
```

## 6. Not Using Context Managers

```python
# ❌ Wrong: manual resource management
session = aiohttp.ClientSession()
response = await session.get(url)
await session.close()

# ✅ Correct: use async context manager
async with aiohttp.ClientSession() as session:
    response = await session.get(url)
    # Session automatically closed
```

## 7. Ignoring Exceptions in gather()

```python
# ❌ Wrong: first exception stops all tasks
results = await asyncio.gather(*tasks)

# ✅ Correct: collect all results including exceptions
results = await asyncio.gather(*tasks, return_exceptions=True)
```

## 8. Not Awaiting Tasks

```python
# ❌ Wrong: task runs but not awaited
asyncio.create_task(background_work())
# Task might not complete before program exits

# ✅ Correct: await task or store reference
task = asyncio.create_task(background_work())
# Later...
await task
```
