# Concurrency Patterns

Patterns for managing concurrent async operations.

## Producer-Consumer Pattern

```python
import asyncio
from asyncio import Queue

async def producer(queue: Queue, producer_id: int, num_items: int):
    """Produce items and put them in queue."""
    for i in range(num_items):
        item = f"Item-{producer_id}-{i}"
        await queue.put(item)
        print(f"Producer {producer_id} produced: {item}")
        await asyncio.sleep(0.1)
    await queue.put(None)  # Signal completion

async def consumer(queue: Queue, consumer_id: int):
    """Consume items from queue."""
    while True:
        item = await queue.get()
        if item is None:
            queue.task_done()
            break

        print(f"Consumer {consumer_id} processing: {item}")
        await asyncio.sleep(0.2)
        queue.task_done()

async def producer_consumer_example():
    """Run producer-consumer pattern."""
    queue = Queue(maxsize=10)

    # Create tasks
    producers = [asyncio.create_task(producer(queue, i, 5)) for i in range(2)]
    consumers = [asyncio.create_task(consumer(queue, i)) for i in range(3)]

    # Wait for producers
    await asyncio.gather(*producers)

    # Wait for queue to be empty
    await queue.join()

    # Cancel consumers
    for c in consumers:
        c.cancel()
```

## Worker Pool Pattern

```python
async def worker(name: str, queue: Queue):
    """Worker that processes items from queue."""
    while True:
        item = await queue.get()
        try:
            await process_item(item)
        finally:
            queue.task_done()

async def worker_pool_example(items: List[str], num_workers: int = 5):
    """Process items using worker pool."""
    queue = Queue()

    # Create workers
    workers = [asyncio.create_task(worker(f"Worker-{i}", queue)) for i in range(num_workers)]

    # Add items to queue
    for item in items:
        await queue.put(item)

    # Wait for all items to be processed
    await queue.join()

    # Cancel workers
    for w in workers:
        w.cancel()
```
