# Core Concepts

Deep dive into Python asyncio's core concepts and components.

## 1. Event Loop

The event loop is the heart of asyncio, managing and scheduling asynchronous tasks.

### Key Characteristics

- **Single-threaded cooperative multitasking**: One thread runs multiple tasks
- **Schedules coroutines**: Manages when coroutines execute
- **Handles I/O**: Non-blocking I/O operations
- **Manages callbacks and futures**: Low-level async primitives

### Event Loop Lifecycle

```python
import asyncio

# Get the event loop (Python < 3.10)
loop = asyncio.get_event_loop()

# Run until complete
loop.run_until_complete(my_coroutine())

# Close loop
loop.close()

# Python 3.7+ - Recommended approach
asyncio.run(my_coroutine())  # Creates, runs, and closes loop automatically
```

## 2. Coroutines

Functions defined with `async def` that can be paused and resumed.

### Basic Syntax

```python
async def my_coroutine():
    """A coroutine function."""
    result = await some_async_operation()
    return result
```

### Calling Coroutines

```python
# Wrong - just creates coroutine object
coro = my_coroutine()

# Correct - await the coroutine
result = await my_coroutine()

# Or run at top level
asyncio.run(my_coroutine())
```

### Nested Coroutines

```python
async def fetch_data(url: str):
    await asyncio.sleep(1)
    return {"url": url, "data": "result"}

async def process_data(url: str):
    data = await fetch_data(url)  # Wait for fetch_data
    processed = {"processed": True, **data}
    return processed

async def main():
    result = await process_data("https://api.example.com")
    print(result)

asyncio.run(main())
```

## 3. Tasks

Scheduled coroutines that run concurrently on the event loop.

### Creating Tasks

```python
import asyncio

async def background_task(name: str):
    print(f"{name} started")
    await asyncio.sleep(2)
    print(f"{name} completed")
    return f"Result from {name}"

async def main():
    # Create tasks (Python 3.7+)
    task1 = asyncio.create_task(background_task("Task 1"))
    task2 = asyncio.create_task(background_task("Task 2"))

    # Tasks start running immediately
    print("Tasks created")

    # Wait for both
    result1 = await task1
    result2 = await task2

    print(f"Results: {result1}, {result2}")

asyncio.run(main())
```

### Task vs Coroutine

```python
# Coroutine - doesn't run until awaited
coro = fetch_data("url")

# Task - starts running immediately
task = asyncio.create_task(fetch_data("url"))

# Both can be awaited
result1 = await coro
result2 = await task
```

## 4. Futures

Low-level objects representing eventual results of async operations.

### Basic Future Usage

```python
import asyncio

async def set_future_result(future: asyncio.Future):
    """Set result after delay."""
    await asyncio.sleep(1)
    future.set_result("Future result")

async def main():
    future = asyncio.Future()

    # Schedule task to set result
    asyncio.create_task(set_future_result(future))

    # Wait for result
    result = await future
    print(result)

asyncio.run(main())
```

## 5. Async Context Managers

Resources that support `async with` for proper cleanup.

### Basic Implementation

```python
class AsyncResource:
    async def __aenter__(self):
        """Setup logic."""
        print("Acquiring resource")
        await asyncio.sleep(0.1)
        return self

    async def __aexit__(self, exc_type, exc_val, exc_tb):
        """Cleanup logic."""
        print("Releasing resource")
        await asyncio.sleep(0.1)

async def use_resource():
    async with AsyncResource() as resource:
        print("Using resource")
        await asyncio.sleep(0.5)

asyncio.run(use_resource())
```

### Using contextlib

```python
from contextlib import asynccontextmanager

@asynccontextmanager
async def async_connection():
    """Create async connection context manager."""
    print("Opening connection")
    conn = await create_connection()
    try:
        yield conn
    finally:
        print("Closing connection")
        await conn.close()

async def main():
    async with async_connection() as conn:
        result = await conn.query("SELECT * FROM users")

asyncio.run(main())
```

## 6. Async Iterators

Objects that support `async for` for iterating over async data sources.

### Basic Async Iterator

```python
class AsyncRange:
    """Async iterator that yields numbers."""

    def __init__(self, start: int, end: int):
        self.start = start
        self.end = end
        self.current = start

    def __aiter__(self):
        return self

    async def __anext__(self):
        if self.current >= self.end:
            raise StopAsyncIteration

        await asyncio.sleep(0.1)  # Simulate async work
        value = self.current
        self.current += 1
        return value

async def main():
    async for number in AsyncRange(1, 5):
        print(number)

asyncio.run(main())
```

### Async Generators

```python
async def async_range(start: int, end: int):
    """Async generator that yields numbers."""
    for i in range(start, end):
        await asyncio.sleep(0.1)
        yield i

async def fetch_pages(url: str, max_pages: int):
    """Fetch paginated data."""
    for page in range(1, max_pages + 1):
        await asyncio.sleep(0.2)  # Simulate API call
        yield {
            "page": page,
            "url": f"{url}?page={page}",
            "data": [f"item_{page}_{i}" for i in range(5)]
        }

async def main():
    async for number in async_range(1, 5):
        print(f"Number: {number}")

    async for page_data in fetch_pages("https://api.example.com", 3):
        print(f"Page {page_data['page']}: {len(page_data['data'])} items")

asyncio.run(main())
```

## 7. Awaitables

Any object that can be used with `await`:

1. **Coroutines**: Functions defined with `async def`
2. **Tasks**: Wrapped coroutines scheduled on event loop
3. **Futures**: Low-level async result containers

```python
import asyncio

# All are awaitables
async def coroutine_func():
    return "coroutine result"

async def main():
    # Awaiting coroutine
    result1 = await coroutine_func()

    # Awaiting task
    task = asyncio.create_task(coroutine_func())
    result2 = await task

    # Awaiting future
    future = asyncio.Future()
    future.set_result("future result")
    result3 = await future

    print(result1, result2, result3)

asyncio.run(main())
```
